/* xenv - expand environment variables in input files  -*- c -*-
   Copyright (C) 2021-2025 Sergey Poznyakoff

   Xenv is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3 of the License, or (at your
   option) any later version.

   Xenv is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with xenv. If not, see <http://www.gnu.org/licenses/>. */
%{
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <sysexits.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <poll.h>
#include <signal.h>
#include "xenv.h"

char const *progname;       /* Program name */
dev_t device;               /* Device number of the filename. */
ino_t inode;                /* Inode number of the filename. */
int undef_error_option;     /* Treat undefined variables as error. */
int synclines_option;       /* Generate `#line NUM "FILE"' lines. */
int retain_unexpanded_option;/* Retain unexpanded constructs in the output. */
unsigned command_timeout = 0;  /* Timeout for command substitution. */
int status = 0;             /* Exit status */
char *env_prefix;           /* -e PREFIX */
static int env_prefix_len;  /* Number of prefix characters collected this
			       far */

static int bracecount = 0;  /* Curly brace nesting level. */

struct point {
	char const *file;
	int line;
	int column;
};

struct locus {
	struct point beg, end;
};

static void push_state_at_point(int newstate, struct point const *point);
static void push_state(int newstate);
static void pop_state(void);
static struct point const *start_point(void);

static char *findenv(char const *ident, int len);
static void expandenv(char const *ident, int len);

enum {  /* Variable expansion modifiers. */
	EXP_DEFVAL = '-',   /* Use default value. */
	EXP_ASSIGN = '=',   /* Assign default value. */
	EXP_ERROR  = '?',   /* Display error if null or unset. */
	EXP_ALTER  = '+',   /* Use alternate value. */
	EXP_TERNARY = '|',  /* Ternary operator. */
	EXP_SPREF  = '#',   /* Remove shortest prefix. */
	EXP_SSUF   = '%',   /* Remove shortest suffix. */

	EXP_LPREF = 256,    /* Remove longest prefix. */
	EXP_LSUF,           /* Remove longest suffix. */
	EXP_PAT,            /* Pattern in ${VAR/PAT/SUBST} */
	EXP_SUBST,          /* Substitutio string in it. */
};

static void expand_inline_push(int type, char *varname, int varlen,
			       int suppress);
static int expand_inline_pop(void);
static void expand_inline_flop(void);

static int exp_subst_upgrade(void);

static int cond_peek(struct point *);
static void cond_push(int result);
static int cond_pop(void);

static void push_input(char const *name, int oknotfound, struct locus *loc);

static int parse_line_directive(char *text);

static void syncline(int);
static void syncline_flush(void);
static void syncline_print(struct point *pt);

static inline int
c_isblank(int c)
{
	return c == ' ' || c == '\t';
}

static inline int
c_isspace(int c)
{
	return c_isblank(c) || c == '\n';
}

static inline int
c_isalpha(int c)
{
	return ('a' <= c && c <= 'z') || ('A' <= c && c <= 'Z');
}

static inline int
c_isdigit(int c)
{
	return '0' <= c && c <= '9';
}

static inline int
c_isalnum(int c)
{
	return c_isalpha(c) || c_isdigit(c);
}

static int beginning_of_line = 1;

static inline void
echo(char const *text, size_t len)
{
	if (len == 0) return;
	if (synclines_option)
		syncline_flush();
	if (len == 1)
		fputc(*text, yyout);
	else
		fwrite(text, len, 1, yyout);
	beginning_of_line = text[len-1] == '\n';
}

#define ECHO echo(yytext, yyleng)

typedef struct hash_entry {
	char *name;
	size_t namelen;
	struct hash_entry *coll;
	void *data;
} HASH_ENTRY;

typedef struct hash_table {
	size_t hash_size;
	size_t entry_count;
	struct hash_entry **entries;
} HASH_TABLE;

struct hash_table_iterator {
	struct hash_table *htab;
	size_t pos;
	struct hash_entry *next;
};

struct hash_table *hash_table_alloc(size_t count);
struct hash_entry *hash_table_next(struct hash_table_iterator *itr);
struct hash_entry *hash_table_first(struct hash_table *htab,
				    struct hash_table_iterator **itr);
void hash_table_free(struct hash_table *htab, void (*dfree)(void*));
struct hash_entry *hash_table_find(struct hash_table *htab, const char *name,
				   size_t namelen, int insert);
static inline struct hash_entry *
hash_table_string_find(struct hash_table *htab, const char *name, int insert)
{
	return hash_table_find(htab, name, strlen(name), insert);
}
void hash_table_delete(struct hash_table *htab, const char *name,
		       size_t namelen);
static inline void
hash_table_string_delete(struct hash_table *htab, const char *name)
{
	return hash_table_delete(htab, name, strlen(name));
}

typedef	struct macro MACRO;
static MACRO *macro_get(char const *name, int insert);

static void divert_start(char const *name, size_t len);
static void divert_stop(void);
static void undivert(char const *name, size_t len);
static void evaldivert(char const *name, size_t len);
static void dropdivert(char const *name, size_t len);

struct filename {
	struct filename *next;
	size_t len;
	char name[1];
};

static struct filename *filename_head;

static char const *
add_filename(char const *name, size_t len)
{
	struct filename *f;
	for (f = filename_head; f; f = f->next)
		if (f->len == len && memcmp(f->name, name, len) == 0)
			return f->name;
	f = xmalloc(sizeof(*f) + len);
	memcpy(f->name, name, len);
	f->name[len] = 0;
	f->len = len;
	f->next = filename_head;
	filename_head = f;
	return f->name;
}

static void
free_filenames(void)
{
	while (filename_head) {
		struct filename *next = filename_head->next;
		free(filename_head);
		filename_head = next;
	}
}

static inline void
point_init(struct point *point, char const *name, size_t len)
{
	point->file = add_filename(name, len);
	point->line = 1;
	point->column = 0;
}

static inline void
point_advance_line(struct point *point, int n)
{
	point->line += n;
	point->column = 0;
}

static void
point_advance_text(struct point *point, char const *text, size_t len)
{
	while (len) {
		char *p = memchr(text, '\n', len);
		if (p == NULL)
			break;
		point_advance_line(point, 1);
		len -= p - text + 1;
		text = p + 1;
	}
	point->column += len;
}

static struct point curpoint;
static struct locus curlocus;

#define YY_USER_ACTION do {						\
		curpoint.column += yyleng;				\
		curlocus.end = curpoint;				\
	} while (0);

static void
point_print(FILE *fp, struct point const *pt)
{
	if (pt->column == 0)
		fprintf(fp, "%s:%u",
			pt->file,
			pt->line);
	else
		fprintf(fp, "%s:%u.%u",
			pt->file,
			pt->line,
			pt->column);
}

static void
locus_print(FILE *fp, struct locus const *loc)
{
	if (loc->beg.column == 0)
		fprintf(fp, "%s:%u",
			loc->beg.file,
			loc->beg.line);
	else if (strcmp(loc->beg.file, loc->end.file))
		fprintf(fp, "%s:%u.%u-%s:%u.%u",
			loc->beg.file,
			loc->beg.line, loc->beg.column,
			loc->end.file,
			loc->end.line, loc->end.column);
	else if (loc->beg.line != loc->end.line)
		fprintf(fp, "%s:%u.%u-%u.%u",
			loc->beg.file,
			loc->beg.line, loc->beg.column,
			loc->end.line, loc->end.column);
	else if (loc->beg.column != loc->end.column)
		fprintf(fp, "%s:%u.%u-%u",
			loc->beg.file,
			loc->beg.line, loc->beg.column,
			loc->end.column);
	else
		fprintf(fp, "%s:%u.%u",
			loc->beg.file,
			loc->beg.line,
			loc->beg.column);
}

static void
error_at_point(struct point const *pt, char const *fmt, ...)
{
	va_list ap;
	point_print(stderr, pt);
	fprintf(stderr, ": ");
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fputc('\n', stderr);
}

static void
error_at_locus(struct locus const *loc, char const *fmt, ...)
{
	va_list ap;
	if (!loc)
		loc = &curlocus;
	locus_print(stderr, loc);
	fprintf(stderr, ": ");
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fputc('\n', stderr);
}

void
xenv_error(int status, int errnum, char const *fmt, ...)
{
	va_list ap;
	fprintf(stderr, "%s: ", progname);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	if (errnum)
		fprintf(stderr, ": %s", strerror(errnum));
	fputc('\n', stderr);
	if (status)
		_exit(status);
}

/*
 * TEXT is a portion of preprocessor statement after the $$ marker.
 * Find the variable identifier part.  Return a pointer to the identifier
 * and store its length in LEN.
 * If KW is supplied, initialize it with the pointer to the preprocessor
 * keyword (after $$ and optional whitespace).
 */
static char *
find_ident(char *text, int *len, char **kw, struct locus *loc)
{
	char *start = text;
	if (text[0] == ' ' || text[0] == '\t')
		text += strspn(text, " \t");
	if (kw)
		*kw = text;
	text += strcspn(text, " \t");
	text += strspn(text, " \t");
	*len = strcspn(text, " \t\n");
	if (loc) {
		loc->beg = curlocus.beg;
		loc->beg.column += 2 + text - start;
		loc->end = loc->beg;
		loc->end.column += *len - 1;
	}
	return text;
}

/*
 * Memory management with error handling.
 */
void
enomem(void)
{
	xenv_error(0, 0, "not enough memory");
	abort();
}

void *
xmalloc(size_t size)
{
	char *p = malloc(size);
	if (!p)
		enomem();
	return p;
}

void *
xcalloc(size_t nmemb, size_t size)
{
	char *p = calloc(nmemb, size);
	if (!p)
		enomem();
	return p;
}

void *
xrealloc(void *ptr, size_t size)
{
	void *p = realloc(ptr, size);
	if (!p)
		enomem();
	return p;
}

void *
x2nrealloc(void *p, size_t *pn, size_t s)
{
	size_t n = *pn;
	char *newp;

	if (!p) {
		if (!n) {
			/* The approximate size to use for initial small
			   allocation requests, when the invoking code
			   specifies an old size of zero.  64 bytes is
			   the largest "small" request for the
			   GNU C library malloc.  */
			enum { DEFAULT_MXFAST = 64 };

			n = DEFAULT_MXFAST / s;
			n += !n;
		}
	} else if ((size_t) -1 / 3 * 2 / s <= n)
		enomem();
	else
		n += (n + 1) / 2;

	newp = realloc(p, n * s);
	if (!newp)
		enomem();
	*pn = n;
	return newp;
}

char *
xstrdup(const char *s)
{
	char *p = xmalloc(strlen(s) + 1);
	strcpy(p, s);
	return p;
}

static HASH_TABLE *bool_tab;

static void
defbool(int enabled, char const *str)
{
	hash_table_free(bool_tab, NULL);
	if (!enabled)
		return;
	if (!str)
		xenv_error(EX_USAGE, 0, "-Wbooleans missing value");
	bool_tab = hash_table_alloc(16);

	while (*str) {
		char const *next;
		char const *q;
		size_t len, n;
		HASH_ENTRY *ent;

		if ((next = strchr(str, ',')) != NULL) {
			len = next - str;
			next++;
		} else {
			len = strlen(str);
			next = str + len;
		}

		if ((q = memchr(str, '/', len)) == NULL)
			xenv_error(EX_USAGE, 0, "bad booleans value near %s", str);

		if (q != str) {
			ent = hash_table_find(bool_tab, str, q - str, 1);
			ent->data = (void*)1;
		}
		q++;
		if ((n = len - (q - str)) > 0) {
			ent = hash_table_find(bool_tab, q, n, 1);
			ent->data = NULL;
		}

		str = next;
	}
}

static int
evalbool(char *val)
{
	HASH_ENTRY *ent;

	if (!val || !bool_tab)
		return 0;
	if ((ent = hash_table_string_find(bool_tab, val, 0)) == NULL)
		return -1;
	return ent->data != NULL;
}

static void
stashsize(int enabled, char const *str)
{
#ifdef HAVE_FOPENCOOKIE
	long n;
	size_t s;
	char *p;

	if (!enabled)
		return;

	if (!str)
		xenv_error(EX_USAGE, 0, "-Wstashsize missing value");
	errno = 0;
	n = strtol(str, &p, 10);
	if (errno || n < 0)
		goto err;

	s = n;
	switch (*p) {
	case 0:
		break;
	case 'g':
	case 'G':
		if ((size_t)-1 / s < 1024)
			goto err;
		s *= 1024;
	case 'm':
	case 'M':
		if ((size_t)-1 / s < 1024)
			goto err;
		s *= 1024;
	case 'k':
	case 'K':
		if ((size_t)-1 / s < 1024)
			goto err;
		s *= 1024;
		break;
	default:
	err:
		xenv_error(EX_USAGE, 0, "bad value for -Wstashsize");
	}
	max_stash_memory = s;
#else
	xenv_error(EX_USAGE, 0,
		   "stash (-Wstashsize) is not available on this system");
#endif
}

static void
stashfiles(int enabled, char const *str)
{
#ifdef HAVE_FOPENCOOKIE
	long n;
	char *p;
	struct rlimit rlim;

	if (!enabled)
		return;
	if (!str)
		xenv_error(EX_USAGE, 0, "-Wstashfiles missing value");
	errno = 0;
	n = strtol(str, &p, 10);
	if (errno || n < 0 || *p)
		xenv_error(EX_USAGE, 0, "bad value for -Wstashfiles");

	if (getrlimit(RLIMIT_NOFILE, &rlim))
		xenv_error(EX_OSERR, errno, "can't get open file limit");

	if (n > rlim.rlim_cur)
		xenv_error(EX_USAGE, 0, "-Wstashfiles is out of range");
	max_stash_files = n;
#else
	xenv_error(EX_USAGE, 0,
		   "stash (-Wstasfiles) is not available on this system");
#endif
}


struct xenv_feature {
	char *name;
	char *descr;
	int enabled;
	void (*setfn)(int, char const *);
};

enum {
	FEATURE_COMMAND,
	FEATURE_COMMENT,
	FEATURE_DIRECTIVE,
	FEATURE_QUOTE,
	FEATURE_ESCAPE,
	FEATURE_MINIMAL,
	FEATURE_BOOLEANS,
	FEATURE_STASHSIZE,
	FEATURE_STASHFILES,
	FEATURE_PARANOID_MEMFREE,
	FEATURE_RPE
};

enum {
	F_OFF,
	F_ON,
	F_DFL
};

static struct xenv_feature feature[] = {
	{
		.name = "command",
		.descr = "enable command expansion",
		.enabled = F_DFL
	},
	{
		.name = "comment",
		.descr = "enable ${* *} comments",
		.enabled = F_DFL
	},
	{
		.name = "directive",
		.descr = "enable xenv directives",
		.enabled = F_DFL
	},
	{
		.name = "quote",
		.descr = "enable $[ ] quotation construct",
		.enabled = F_DFL
	},
	{
		.name = "escape",
		.descr = "enable backslash escape",
		.enabled = F_DFL
	},
	{
		.name = "minimal",
		.descr = "disable most features, assume -e ENV",
		.enabled = F_OFF
	},
	{
		.name = "booleans",
		.descr = "define boolean values",
		.enabled = F_DFL,
		.setfn = defbool
	},
	{
		.name = "stashsize",
		.descr = "maximum stash memory size",
		.enabled = F_DFL,
		.setfn = stashsize
	},
	{
		.name = "stashfiles",
		.descr = "number of open stash files allowed",
		.enabled = F_DFL,
		.setfn = stashfiles
	},
	{
		.name = "paranoid-memfree",
		.descr = "free all used memory before exiting",
		.enabled = F_OFF
	},
	{
		.name = "rpe",
		.descr = "retain partially expanded constructs on output",
		.enabled = F_OFF
	},
	{ NULL },
};

static int
feature_set(char const *name)
{
	int i, enabled = F_ON;
	size_t len;
	char *val;

	if ((val = strchr(name, '=')) != NULL) {
		len = val - name;
		val++;
	} else
		len = strlen(name);

	if (val == NULL && strncmp(name, "no-", 3) == 0) {
		name += 3;
		len -= 3;
		enabled = F_OFF;
	}

	if (*name) {
		for (i = 0; feature[i].name; i++) {
			if (strlen(feature[i].name) == len &&
			    memcmp(feature[i].name, name, len) == 0) {
				if (feature[i].setfn)
					feature[i].setfn(enabled, val);
				else if (val)
					break;
				feature[i].enabled = enabled;
				return 0;
			}
		}
	}
	return -1;
}

static void
feature_set_minimal(void)
{
	int i;
	for (i = 0; feature[i].name; i++) {
		if (i != FEATURE_MINIMAL && feature[i].enabled == F_DFL)
			feature[i].enabled = F_OFF;
	}
}

static inline int
feature_is_set(int f)
{
	return feature[f].enabled;
}

static void expandcom(char *str, struct locus const *locus);
static int testcom(char *str, size_t len, struct locus const *locus);

static int collect_delimited_text(int od, int cd,
				  void (*collector)(void *, int),
				  void *closure);
static void ignore_text(void);

static void
collect_echo(void *closure, int c)
{
	fputc(c, yyout);
}

struct stringbuf {
	char *text;
	size_t len;
	size_t size;
};

#define STRINGBUF_INITIALIZER { NULL, 0, 0 }

static inline void
stringbuf_init(struct stringbuf *acc)
{
	memset(acc, 0, sizeof(*acc));
}

static inline void
stringbuf_reset(struct stringbuf *acc)
{
	acc->len = 0;
}

static inline void
stringbuf_add(struct stringbuf *acc, int c)
{
	if (acc->len == acc->size)
		acc->text = x2nrealloc(acc->text, &acc->size,
				      sizeof(acc->text[0]));
	acc->text[acc->len++] = c;
}

static inline void
stringbuf_free(struct stringbuf *acc)
{
	free(acc->text);
}

static inline void
stringbuf_revert(struct stringbuf *acc)
{
	int i, j;
	for (i = 0, j = acc->len-1; j > i; i++, j--) {
		char c = acc->text[i];
		acc->text[i] = acc->text[j];
		acc->text[j] = c;
	}
}

static void
collect_text(void *closure, int c)
{
	stringbuf_add(closure, c);
}

/*
 * General-purpose stack.
 */
typedef struct xenv_stack {
	size_t entry_size;
	char *stack;
	size_t max_stack;
	size_t tos;
} XENV_STACK;

#define XENV_STACK_INITIALIZER(type)  { sizeof(type), NULL, 0, 0 }
#define XENV_INIT_SIZE 64

static void
xenv_stack_free(XENV_STACK *stk)
{
	free(stk->stack);
}

static inline int
xenv_stack_size(XENV_STACK *stk)
{
	return stk->tos;
}

static void *
xenv_stack_peek(XENV_STACK *stk, int dec)
{
	if (-dec > stk->tos)
		return NULL;
	return stk->stack + stk->entry_size * (stk->tos + dec);
}

static void *
xenv_stack_push(XENV_STACK *stk)
{
	if (stk->tos == stk->max_stack) {
		if (!stk->stack)
			stk->stack = xcalloc(stk->max_stack = XENV_INIT_SIZE,
					     stk->entry_size);
		else
			stk->stack = x2nrealloc(stk->stack, &stk->max_stack,
						stk->entry_size);
	}
	++stk->tos;
	return xenv_stack_peek(stk, -1);
}

static void *
xenv_stack_pop(XENV_STACK *stk)
{
	if (stk->tos == 0) {
		error_at_point(&curpoint, "out of popup space");
		exit(EX_SOFTWARE);
	}
	--stk->tos;
	return xenv_stack_peek(stk, 0);
}

static void loop_begin(void);
static void paramctx_collect(void);
static int loop_next(void);
static void paramctx_close(void);
static void paramctx_init(int type,
			  char const *name, size_t nlen,
			  char const *prefix, size_t plen);
static int paramctx_type(void);
static void paramctx_setvar(void);
static void paramctx_callmacro(void);
static void paramctx_recover(void);

enum paramctx_type {
	PARAMCTX_NONE = -1,
	PARAMCTX_RECOVERY,
	PARAMCTX_FOREACH,
	PARAMCTX_RANGE,
	PARAMCTX_EVAL,
	PARAMCTX_SET,
	PARAMCTX_DEFMACRO,
	PARAMCTX_CALLMACRO
};

/* Expected number of $$end statements that should be seen before the one
   that finishes the COLLECT state. */
static unsigned endcnt;

struct command_collect {
	FILE *saved_yyout;
	int negate;
	struct locus locus;
};

static struct command_collect command_collect;

static void
command_collect_begin(struct locus const *loc, int negate, int suppress)
{
	command_collect.saved_yyout = yyout;
	command_collect.locus = *loc;
	command_collect.negate = negate;
	if (suppress) {
		if ((yyout = sink_open()) == NULL)
			xenv_error(EX_OSERR, errno, "can't open sink");
	} else if ((yyout = stash_open()) == NULL) {
		xenv_error(EX_OSERR, errno, "can't open stash");
	}
}

static void
command_collect_update_line(const char *text, size_t len)
{
	fwrite(text, len, 1, yyout);
	point_advance_line(&command_collect.locus.end, 1);
}

static char *
command_collect_finish(size_t *len, struct locus *loc, int *negate)
{
	size_t n;
	off_t size = ftell(yyout);
	char *buf = xmalloc(size + 1);
	FILE *fp = yyout;
	yyout = command_collect.saved_yyout;
	rewind(fp);
	if ((n =fread(buf, 1, size, fp)) != size)
		xenv_error(EX_OSERR, errno, "read error composing command (%zu)", n);
	buf[size] = 0;
	fclose(fp);
	*len = size;
	*loc = command_collect.locus;
	*negate = command_collect.negate;
	memset(&command_collect, 0, sizeof(command_collect));
	return buf;
}

static int
vercmp(char const *req, char const *ver)
{
	while (*req) {
		long rn, vn;
		errno = 0;
		rn = strtol(req, (char**) &req, 10);
		if (errno || (*req && *req != '.'))
			return -2;
		errno = 0;
		vn = strtol(ver, (char**) &ver, 10);
		if (errno || (*ver && *ver != '.'))
			return -2;
		if (vn < rn)
			return -1;
		if (vn > rn)
			return 1;
		if (*req == 0 || *ver == 0)
			break;
		ver++;
		req++;
	}
	return 0;
}
%}
  /* Identifier */
IDENT [a-zA-Z_][a-zA-Z_0-9]*
  /* Whitespace */
WS [ \t][ \t]*
  /* Optional whitespace */
OWS [ \t]*

  /*
   * States
   */
%x COMMENT FALSE SIGIL DSIGIL VERBATIM ALT DQ DQS PREFIX PARAMS
%x COLLECT COM FELSE PAT
  /*
   * Possible state transitions are:
   *
   *  State     Input          New state         Comment
   *  --------  ------------   ---------------   -------
   *   INITIAL  $              SIGIL             Unless -e is used
   *   INITIAL  $              PREFIX            If -e is used
   *   INITIAL  $$             DSIGIL            Only at the beginning of
   *                                             line, optionally preceded
   *                                             by whitespace.
   *                                             Disabled by -Wno-directive.
   *   PREFIX   env_prefix     SIGIL
   *   SIGIL    {<ID>          ALT               ID stands for an identifier
   *                                             followed by optional ?
   *                                             and one of -, =, ?, +, |
   *   SIGIL    {<TRIM>        PAT               TRIM stands for an identifier
   *                                             followed by #, ##, %, or %%.
   *   SIGIL    {*             COMMENT           Disabled by -Wno-comment
   *   ALT      '"'            DQ
   *   DQ       $              SIGIL
   *   COMMENT  *}             INITIAL
   *   DSIGIL   "verbatim"     VERBATIM          Unless prior state was FALSE.
   *   DSIGIL   "set IDENT$"   INITIAL
   *            'set IDENT'    PARAMS
   *            "loop IDENT"   PARAMS
   *            "ifcom".*\\n   COM               Unless \ is escaped
   *            "ifncom".*\\n  COM               --"--
   *   DSIGIL,FELSE
   *            "ifset"        INITIAL or FALSE  Depending on condition.
   *            "ifnset"                         --"--
   *            "ifdef"                          --"--
   *            "ifndef"                         --"--
   *            "ifcom"                          --"--
   *            "ifncom"                         --"--
   *            "else"                           --"--
   *   FELSE    "endif"        INITIAL           state popped
   *   PARAMS "\n"           COLLECT
   *   COLLECT  "end"          INITIAL           starts first loop iteration
   *   VERBATIM "end"          INITIAL
   *   FALSE    "$$"           FELSE
   *   COM      \n             prev              pop state
   *                                             unless \n is preceded by
   *                                             an unescaped backslash
   */
%%
<INITIAL,ALT,PAT,DQ,DQS>{
	\n+  {
		point_advance_line(&curpoint, yyleng);
		ECHO;
	}
}

<PARAMS>{
	\'[^\']*\' {
		point_advance_text(&curpoint, yytext, yyleng);
		ECHO;
	}
	\"      ECHO; push_state(DQS);
	\\\n    point_advance_line(&curpoint, 1);
	\n {
		point_advance_line(&curpoint, 1);
		pop_state();
		switch (paramctx_type()) {
		case PARAMCTX_SET:
			paramctx_setvar();
			paramctx_close();
			syncline(0);
			beginning_of_line = 1;
			break;

		case PARAMCTX_CALLMACRO:
			paramctx_callmacro();
			paramctx_close();
			syncline(0);
			beginning_of_line = 1;
			break;

		default:
			push_state(COLLECT);
			paramctx_collect();
		}
	}
}

<ALT,PAT>{
	\{  {
		bracecount++; ECHO;
	}

	\}  {
		if (expand_inline_pop())
			ECHO;
		else
			pop_state();
		bracecount--;
	}
}

<ALT>{
	\|  expand_inline_flop();
}

<PAT>{
	"/" if (exp_subst_upgrade() == -1) ECHO;
	     /* Note: it would suffice to match "\\"[\\/] here.  The characters
		% and # are added for symmetry with their escaped use at the
		beginning of pattern. */
	"\\"[\\#%/] echo(yytext+1, 1);
}

<INITIAL,ALT,PAT,DQ,DQS,PARAMS>{
	\$ {
		curlocus.beg = curpoint;
		if (env_prefix) {
			env_prefix_len = 0;
			push_state(PREFIX);
		} else {
			push_state(SIGIL);
		}
	}
}

<INITIAL>\\[\\\$] {
	/*
	 * Backslash handling in initial state is governed by -W[no-]escape
	 */
	if (feature_is_set(FEATURE_ESCAPE)) {
		fputc(yytext[1], yyout);
		beginning_of_line = 0;
	} else {
		yyless(1);
		ECHO;
	}
}

<ALT,PAT,DQ,DQS,PARAMS>\\[\\\$] {
	/*
	 * In expansion states, backslash is always escaping another backslash
	 * and dollar sign.
	 */
	fputc(yytext[1], yyout);
	beginning_of_line = 0;
}

<INITIAL,ALT>{
	^"#line ".*\n {
		if (parse_line_directive(yytext)) {
			point_advance_line(&curpoint, 1);
		}
		syncline(0);
		beginning_of_line = 1;
	}

	^{OWS}"$$" {
		if (feature_is_set(FEATURE_DIRECTIVE)) {
			curlocus.beg = curpoint;
			/* Column points to the second '$'; adjust it. */
			curlocus.beg.column--;
			push_state(DSIGIL);
		} else {
			ECHO;
		}
	}
}

<INITIAL><<EOF>> {
	struct point pt;
	if (cond_peek(&pt) != -1) {
		error_at_point(&pt, "end of file in conditional");
		status = EX_DATAERR;
	}
	return 0;
}

<ALT,PAT>{
	\\[\\\'\"] {
		fputc(yytext[1], yyout);
		beginning_of_line = 0;
	}

	\"      push_state(DQ);

	"'"[^\']*"'" {
		fwrite(yytext+1, yyleng-2, 1, yyout);

		curpoint.column -= yyleng;
		point_advance_text(&curpoint, yytext, yyleng);
		curlocus.end = curpoint;
		syncline(0);
	}

	<<EOF>> {
		error_at_point(start_point(),
			       "end of file in variable expansion");
		status = EX_DATAERR;
		return 0;
	}
}

<DQ>{
	\\[\\\'\"] {
		fputc(yytext[1], yyout);
		beginning_of_line = 0;
	}

	\"	pop_state();

	<<EOF>> {
		error_at_point(start_point(),
			       "end of file in variable expansion: "
			       "missing closing double quote");
		status = EX_DATAERR;
		return 0;
	}
}

<DQS>{
	\\[\\\'\"]   ECHO;
	\"	pop_state(); ECHO;

	<<EOF>> {
		error_at_point(start_point(), "missing closing double quote");
		return 0;
	}
}

<COMMENT>{
	[^*\n]*        /* eat anything that's not a '*' */

	"*"+[^*}\n]*   /* eat up '*'s not followed by '}'s */

		       /* Keep track of line numbers. */
	\n             point_advance_line(&curpoint, 1);

	"*"+"}"        { pop_state(); syncline(0); }
	 <<EOF>> {
		error_at_point(start_point(), "end of file in comment");
		status = EX_DATAERR;
		return 0;
	 }
}

<PREFIX>{
	\n|. {
		if (yytext[0] == env_prefix[env_prefix_len]) {
			if (env_prefix[++env_prefix_len] == 0) {
				pop_state();
				push_state(SIGIL);
			}
		} else {
			pop_state();
			fputc('$', yyout);
			beginning_of_line = 0;
			echo(env_prefix, env_prefix_len);
			ECHO;
			if (yytext[0] == '\n')
				point_advance_line(&curpoint, 1);
		}
	}
}

<SIGIL>{
	"{*" {
		pop_state();
		if (feature_is_set(FEATURE_COMMENT)) {
			push_state_at_point(COMMENT, &curlocus.beg);
		} else {
			fputc('$', yyout);
			beginning_of_line = 0;
			ECHO;
		}
	}

	"[" {
		if (feature_is_set(FEATURE_QUOTE)) {
			/*
			 * Save start_point for eventual use in the diagnostic
			 * message below
			 */
			struct point pt = *start_point();
			int c = collect_delimited_text('[', ']', collect_echo,
						       yyout);
			pop_state();
			if (c == EOF) {
				error_at_point(&pt,
					       "end of file in inline"
					       " verbatim text");
				status = EX_DATAERR;
				return 0;
			}
		} else {
			pop_state();
			fputc('$', yyout);
			beginning_of_line = 0;
			ECHO;
		}
	}

	"(" {
		if (feature_is_set(FEATURE_COMMAND)) {
			struct stringbuf t = STRINGBUF_INITIALIZER;
			struct point pt = *start_point();
			int c = collect_delimited_text('(', ')', collect_text,
						       &t);
			pop_state();
			collect_text(&t, 0);
			if (c == EOF) {
				error_at_point(&pt,
					       "end of file in command"
					       " expansion");
				status = EX_DATAERR;
				free(t.text);
				return 0;
			}
			expandcom(t.text, &curlocus);
			free(t.text);
		} else {
			pop_state();
			fputc('$', yyout);
			beginning_of_line = 0;
			ECHO;
		}
	}

	{IDENT} {
		pop_state();
		if (env_prefix) {
			fputc('$', yyout);
			beginning_of_line = 0;
			echo(env_prefix, strlen(env_prefix));
			ECHO;
		} else {
			expandenv(yytext, yyleng);
		}
	}

	\{{IDENT}\} {
		pop_state();
		expandenv(yytext + 1, yyleng - 2);
	}

	\{{IDENT}"/\\"[\\#%/] {
		int len = yyleng-3;
		pop_state();
		bracecount++;
		if (synclines_option)
			syncline_flush();
		expand_inline_push(EXP_PAT, yytext + 1, len - 1, 0);
		echo(yytext + yyleng - 2, 2);
		synclines_option = 0;
		push_state(PAT);
	}

	\{{IDENT}"/""/"?  {
		int len = yyleng-1;
		if (yytext[len-1] == '/')
			len--;
		pop_state();
		bracecount++;
		if (synclines_option)
			syncline_flush();
		expand_inline_push(EXP_PAT, yytext + 1, len - 1, 0);
		if (len != yyleng-1)
			echo(yytext + yyleng - 1, 1);
		synclines_option = 0;
		push_state(PAT);
	}

	\{{IDENT}((#{1,2})|(%{1,2})) {
		int len = yyleng-1;
		int type = yytext[len];

		pop_state();
		if (yytext[len-1] == type) {
			switch (type) {
			case EXP_SPREF:
				type = EXP_LPREF;
				break;

			case EXP_SSUF:
				type = EXP_LSUF;
				break;

			default:
				abort();
			}
			len--;
		}

		bracecount++;
		if (synclines_option)
			syncline_flush();
		expand_inline_push(type, yytext + 1, len - 1, 0);
		synclines_option = 0;
		push_state(PAT);
	}

	\{{IDENT}:?[-=?+|] {
		int len;
		char *val;
		int test_null = 0;
		int type = yytext[yyleng-1];
		int suppress;

		pop_state();
		if (yytext[yyleng-2] == ':') {
			test_null = 1;
			len = yyleng - 3;
		} else
			len = yyleng - 2;
		val = findenv(yytext + 1, len);
		if (val && !(test_null && *val == 0)) {
			if (type == EXP_TERNARY || type == EXP_ALTER) {
				suppress = 0;
			} else {
				fwrite(val, strlen(val), 1, yyout);
				suppress = 1;
			}
		} else if (type == EXP_TERNARY || type == EXP_ALTER) {
			suppress = 1;
		} else {
			suppress = 0;
			if (type == EXP_ERROR ) {
				int c = input();
				if (c == '}') {
					/* Provide default error message */
					if (test_null)
						error_at_locus(&curlocus,
							       "variable %.*s"
							       " null or not"
							       " set",
							       len,
							       yytext + 1);
					else
						error_at_locus(&curlocus,
							       "variable %.*s"
							       " not set",
							       len,
							       yytext + 1);
					YY_BREAK;
				}
				unput(c);
			}
		}
		bracecount++;
		expand_inline_push(type, yytext + 1, len, suppress);
		push_state(ALT);
	}

	\n|.   {
		char s[2];
		s[0] = '$';
		s[1] = yytext[0];
		echo(s, 2);
		pop_state();
		if (yytext[0] == '\n')
			point_advance_line(&curpoint, 1);
	}

	<<EOF>> {
		fputc('$', yyout);
		return 0;
	}
}

<DSIGIL>{
	{OWS}"dnl"({WS}[^\n]*)?\n {
		pop_state();
		point_advance_line(&curpoint, 1);
		syncline(0);
	}
	{OWS}"ignore"({WS}.*)?\n {
		pop_state();
		point_advance_line(&curpoint, 1);
		ignore_text();
	}
	{OWS}"require"{WS}([0-9]+)(\.[0-9]+){0,2}{OWS}"\n" {
		int len;
		char *version = find_ident(yytext, &len, NULL, NULL);

		pop_state();
		version[len] = 0;

		if (vercmp(version, PACKAGE_VERSION) < 0) {
			error_at_locus(&curlocus,
				       "required version mismatch: "
				       "requested at least %s, "
				       "but have %s",
				       version, PACKAGE_VERSION);
			exit(EX_CONFIG);
		}
		point_advance_line(&curpoint, 1);
	}
	{OWS}"verbatim"{OWS}"\n" {
		pop_state();
		push_state_at_point(VERBATIM, &curlocus.beg);
		point_advance_line(&curpoint, 1);
	}

	{OWS}"unset"{WS}{IDENT}{OWS}\n {
		int len;
		char *ident = find_ident(yytext, &len, NULL, NULL);

		ident[len] = 0;
		unsetenv(ident);

		point_advance_line(&curpoint, 1);
		syncline(0);
		pop_state();
	}

	{OWS}"set"{WS}{IDENT} {
		int len;
		char *ident = find_ident(yytext, &len, NULL, NULL);
		paramctx_init(PARAMCTX_SET, ident, len, yytext, yyleng);
		pop_state();
		syncline(0);
		push_state(PARAMS);
	}

	{OWS}((def)|(exp))macro{WS}{IDENT} {
		int len;
		char *kw;
		char *ident = find_ident(yytext, &len, &kw, NULL);
		paramctx_init(kw[0] == 'd'
				 ? PARAMCTX_DEFMACRO
				 : PARAMCTX_CALLMACRO,
			      ident, len,
			      yytext, yyleng);
		pop_state();
		syncline(0);
		push_state(PARAMS);
	}

	{OWS}"source"{WS}.*\n |
	{OWS}"include"{WS}.*\n {
		char *filename;
		int namelen;
		struct locus loc;

		pop_state();
		filename = find_ident(yytext, &namelen, NULL, &loc);
		filename[namelen] = 0;
		point_advance_line(&curpoint, 1);
		push_input(filename, 0, &loc);
		syncline(0);
	}

	{OWS}"sinclude"{WS}.*\n {
		char *filename;
		int namelen;
		struct locus loc;

		pop_state();
		filename = find_ident(yytext, &namelen, NULL, &loc);
		filename[namelen] = 0;
		point_advance_line(&curpoint, 1);
		push_input(filename, 1, &loc);
		syncline(0);
	}

	{OWS}"divert"{WS}{IDENT}{OWS}\n {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		point_advance_line(&curpoint, 1);
		name = find_ident(yytext, &namelen, NULL, &loc);
		divert_start(name, namelen);
		syncline(0);
	}

	{OWS}"divert"{OWS}\n {
		pop_state();
		point_advance_line(&curpoint, 1);
		divert_stop();
		syncline(0);
	}

	{OWS}"undivert"{WS}{IDENT}{OWS}\n {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		name = find_ident(yytext, &namelen, NULL, &loc);
		divert_stop();
		undivert(name, namelen);
		point_advance_line(&curpoint, 1);
	}

	{OWS}"evaldivert"{WS}{IDENT}{OWS}\n {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		name = find_ident(yytext, &namelen, NULL, &loc);
		divert_stop();
		evaldivert(name, namelen);
		point_advance_line(&curpoint, 1);
	}

	{OWS}"dropdivert"{WS}{IDENT}{OWS}\n {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		name = find_ident(yytext, &namelen, NULL, &loc);
		dropdivert(name, namelen);
		point_advance_line(&curpoint, 1);
	}

	{OWS}"loop"{WS}{IDENT}{WS} {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		name = find_ident(yytext, &namelen, NULL, &loc);
		paramctx_init(PARAMCTX_FOREACH, name, namelen, yytext, yyleng);
		push_state(PARAMS);
	}

	{OWS}"range"{WS}{IDENT}{WS} {
		char *name;
		int namelen;
		struct locus loc;

		pop_state();
		name = find_ident(yytext, &namelen, NULL, &loc);
		paramctx_init(PARAMCTX_RANGE, name, namelen, yytext, yyleng);
		push_state(PARAMS);
	}

	{OWS}eval{OWS}\n {
		pop_state();
		point_advance_line(&curpoint, 1);
		paramctx_init(PARAMCTX_EVAL, NULL, 0, yytext, yyleng);
	}
	{OWS}end{OWS}\n {
		pop_state();
		if (paramctx_type() == PARAMCTX_EVAL) {
			loop_begin();
			point_advance_line(&curpoint, 1);
		} else {
			error_at_locus(&curlocus, "unmatched $$end");
			status = EX_DATAERR;
			if (feature_is_set(FEATURE_RPE)) {
				echo("$$", 2);
				ECHO;
			}
		}
	}

	{OWS}((warning)|(error)){WS}.*\n {
		int len;
		char *msg, *kw;

		pop_state();
		msg = find_ident(yytext, &len, &kw, NULL);
		len = strlen(msg);
		while (len > 0 && c_isspace(msg[len-1]))
		       --len;
		if (*kw == 'w')
			error_at_locus(&curlocus, "warning: %*.*s",
				       len, len, msg);
		else {
			error_at_locus(&curlocus, "error: %*.*s",
				       len, len, msg);
			status = EX_DATAERR;
		}
		point_advance_line(&curpoint, 1);
	}
	{OWS}exit{OWS}\n {
		return 0;
	}
	{OWS}exit{WS}.*\n {
		int len;
		char *str;
		long n;

		pop_state();
		str = find_ident(yytext, &len, NULL, NULL);
		n = strtol(str, NULL, 10);
		if (n > 127) {
			curlocus.beg.column += str - yytext + 2;
			error_at_locus(&curlocus, "exit code out of range");
			n = EX_DATAERR;
		}
		status = n;
		return 0;
	}

	{OWS}((exp|def)macro|d(ivert|nl|ropdivert)|e(lse|nd(if)?|rror|val(divert)?|xit)|i(f(com|def|false|n(com|def|set)|set|true)|(gnor|nclud)e)|loop|r((ang|equir)e)|s(et|(includ|ourc)e)|un((diver|se)t)|verbatim|warning) {
		char s[2];

		pop_state();
		error_at_locus(&curlocus, "malformed $$%s directive", yytext);
		status = EX_DATAERR;
		s[0] = s[1] = '$';
		echo(s, 2);
		ECHO;
	}

	{OWS}{IDENT} {
		char *ident = yytext + strspn(yytext, " \t");

		pop_state();
		if (macro_get(ident, 0)) {
			paramctx_init(PARAMCTX_CALLMACRO,
				      ident, strlen(ident),
				      yytext, yyleng);
			syncline(0);
			push_state(PARAMS);
		} else {
			char s[2];

			error_at_locus(&curlocus,
				       "unrecognized directive: $$%s",
				       yytext);
			status = EX_DATAERR;
			s[0] = s[1] = '$';
			echo(s, 2);
			ECHO;
		}
	}

	. {
		char s[3];

		pop_state();
		error_at_locus(&curlocus, "stray character (%#o) after $$",
			       yytext[0]);
		status = EX_DATAERR;
		s[0] = s[1] = '$';
		s[2] = yytext[0];
		echo(s, 3);
	}

	<<EOF>> {
		fputs("$$", yyout);
		return 0;
	}
}

<DSIGIL,FELSE>{
	{OWS}"if"n?(def|set){WS}{IDENT}{OWS}\n {
		int result;

		pop_state();
		result = cond_peek(NULL);
		if (result) {
			int len;
			char *ident;
			char *val;
			char *kw;
			int neg;

			ident = find_ident(yytext, &len, &kw, NULL);
			neg = kw[2] == 'n';
			val = findenv(ident, len);
			result = val != NULL &&
				(kw[2+neg] == 'd' || *val != 0);
			if (neg)
				result = !result;
		}
		cond_push(result);
		point_advance_line(&curpoint, 1);
	}

	{OWS}if(true|false){WS}{IDENT}{OWS}\n {
		int result;

		pop_state();
		result = cond_peek(NULL);
		if (result) {
			int len;
			char *ident, *kw;
			struct locus loc;

			ident = find_ident(yytext, &len, &kw, &loc);
			if ((result = evalbool(findenv(ident, len))) == -1) {
				error_at_locus(&loc,
					       "variable neither true nor"
					       " false");
				status = EX_DATAERR;
				result = 0;
			} else if (kw[2] == 'f') { //iffalse
				result = ! result;
			}
		}
		cond_push(result);
		point_advance_line(&curpoint, 1);
	}

	{OWS}ifn?com{WS}(.*[^\\])?\\\n {
		int len;
		char *ident;
		struct locus loc;
		char *kw;

		ident = find_ident(yytext, &len, &kw, &loc);
		/* Recompute len to be full command len */
		len = yyleng - (ident - yytext) - 2;
		/* Adjust end point accordingly */
		loc.end.column = loc.beg.column + len - 1;

		pop_state();
		push_state(COM);
		command_collect_begin(&loc, kw[2] == 'n', !cond_peek(NULL));
		command_collect_update_line(ident, len);
		point_advance_line(&curpoint, 1);
	}

	{OWS}ifn?com{WS}.*\n {
		int result;

		pop_state();
		result = cond_peek(NULL);
		if (result) {
			int len;
			char *ident;
			struct locus loc;
			char *kw;

			ident = find_ident(yytext, &len, &kw, &loc);
			/* Recompute len to be full command len */
			len = yyleng - (ident - yytext) - 1;
			/* Adjust end point accordingly */
			loc.end.column = loc.beg.column + len - 1;
			result = testcom(ident, len, &loc);
			if (kw[2] == 'n')
				result = !result;
		}
		cond_push(result);
		point_advance_line(&curpoint, 1);
	}

	{OWS}"else"{OWS}\n {
		pop_state();
		if (cond_peek(NULL) == -1) {
			error_at_locus(&curlocus,
				       "$$else without matching $$if");
			status = EX_DATAERR;
			fputc('$', yyout);
			fputc('$', yyout);
			beginning_of_line = 0;
			ECHO;
		} else {
			int result = cond_pop();
			if (cond_peek(NULL))
				result = !result;
			cond_push(result);
		}
		point_advance_line(&curpoint, 1);
	}

	{OWS}"endif"{OWS}\n {
		pop_state();
		if (cond_peek(NULL) == -1) {
			error_at_locus(&curlocus,
				       "$$endif without matching $$if");
			status = EX_DATAERR;
			fputc('$', yyout);
			fputc('$', yyout);
			beginning_of_line = 0;
			ECHO;
		} else {
			cond_pop();
		}
		point_advance_line(&curpoint, 1);
		syncline(0);
	}
}

<FELSE>{
	\n   { point_advance_line(&curpoint, 1); pop_state(); }
	.    pop_state();
	<<EOF>> {
		error_at_point(start_point(), "end of file in conditional");
		status = EX_DATAERR;
		return 0;
	}
}

<COM>{
	(.*[^\\])?\\\n {
		command_collect_update_line(yytext, yyleng-2);
		point_advance_line(&curpoint, 1);
	}

	(.*[^\\])?\n |
	.*\\\\\n {
		int result;

		pop_state();
		result = cond_peek(NULL);
		if (result) {
			char *com;
			size_t len;
			struct locus loc;
			int neg;

			command_collect_update_line(yytext, yyleng-1);
			com = command_collect_finish(&len, &loc, &neg);
			result = testcom(com, len, &loc);
			free(com);
			if (neg)
				result = !result;
		}
		cond_push(result);
		point_advance_line(&curpoint, 1);
	}

	<<EOF>> {
		error_at_point(&curpoint, "end of file in $$ifcom statement");
		status = EX_DATAERR;
		return 0;
	}
}

<FALSE>{
	\n+	point_advance_line(&curpoint, yyleng);

	^{OWS}"$$" {
		curlocus.beg = curpoint;
		curlocus.beg.column--;
		push_state(FELSE);
	}
	. ;

	<<EOF>> {
		error_at_point(start_point(), "end of file in conditional");
		status = EX_DATAERR;
		return 0;
	}
}

<VERBATIM>{
	\n+    {
		ECHO;
		point_advance_line(&curpoint, yyleng);
	}

	^{OWS}"$$"{OWS}"end"{OWS}"\n" {
		point_advance_line(&curpoint, 1);
		pop_state();
	}

	<<EOF>> {
		error_at_point(start_point(), "end of file in verbatim block");
		status = EX_DATAERR;
		return 0;
	}
}

<COLLECT>{
	\n+    {
		ECHO;
		point_advance_line(&curpoint, yyleng);
	}

	^{OWS}"$$"{OWS}(loop|range){WS} {
		endcnt++;
		ECHO;
	}

	^{OWS}"dnl"({WS}[^\n]*)?\n point_advance_line(&curpoint, 1);

	^{OWS}"$$"{OWS}ignore({WS}.*)?\n {
		point_advance_line(&curpoint, 1);
		ignore_text();
	}
	^{OWS}"$$"{OWS}(verbatim|eval){OWS}"\n" {
		point_advance_line(&curpoint, 1);
		endcnt++;
		ECHO;
	}
	^{OWS}"$$"{OWS}"end"{OWS}"\n" {
		if (endcnt == 0) {
			loop_begin();
			pop_state();
		} else {
			ECHO;
			--endcnt;
		}
		point_advance_line(&curpoint, 1);
	}

	<<EOF>> {
		error_at_point(start_point(), "end of file in loop body");
		status = EX_DATAERR;
		if (feature_is_set(FEATURE_RPE)) {
			paramctx_recover();
		}
		paramctx_close();
		return 0;
	}
}
%%

struct hash_table *
hash_table_alloc(size_t count)
{
	struct hash_table *ht = xcalloc(1, sizeof(*ht));
	ht->entries = xcalloc(count, sizeof(ht->entries[0]));
	ht->hash_size = count;
	ht->entry_count = 0;
	return ht;
}

static size_t
hash_string(const char *s, size_t l, size_t n)
{
	size_t v = 0;

	for (; l; l--)
		v = (v * 31 + *(unsigned char*)s++) % n;
	return v;
}

struct hash_entry *
hash_table_next(struct hash_table_iterator *itr)
{
	struct hash_entry *ep;

	if (itr->next == NULL) {
		for (;;) {
			itr->pos++;
			if (itr->pos == itr->htab->hash_size)
				return NULL;
			if ((itr->next = itr->htab->entries[itr->pos]) != NULL)
				break;
		}
	}
	ep = itr->next;
	itr->next = ep->coll;
	return ep;
}

struct hash_entry *
hash_table_first(struct hash_table *htab, struct hash_table_iterator **itr)
{
	*itr = xmalloc(sizeof(**itr));
	(*itr)->htab = htab;
	(*itr)->pos = 0;
	(*itr)->next = htab->entries[(*itr)->pos];
	return hash_table_next(*itr);
}

void
hash_table_free(struct hash_table *htab, void (*dfree)(void*))
{
	struct hash_table_iterator *itr;
	struct hash_entry *ep;

	if (!htab)
		return;
	for (ep = hash_table_first(htab, &itr); ep; ep = hash_table_next(itr)) {
		if (dfree)
			dfree(ep->data);
		free(ep->name);
		free(ep);
	}
	free(itr);
	free(htab->entries);
	free(htab);
}

struct hash_entry *
hash_table_find(struct hash_table *htab, const char *name, size_t namelen,
		int insert)
{
	size_t hv = hash_string(name, namelen, htab->hash_size);
	struct hash_entry **tail = &htab->entries[hv], *head;

	for (;;) {
		head = *tail;
		if (head == NULL)
			break;
		if (namelen == head->namelen && memcmp(head->name, name, namelen) == 0)
			return head;

		tail = &head->coll;
	}

	if (!insert)
		return NULL;

	if (((htab->entry_count + 1) * 100 / htab->hash_size) > 75) {
		struct hash_entry **etab;
		struct hash_table_iterator *itr;
		struct hash_entry *ep;
		size_t n = htab->hash_size;
		if ((size_t) -1 / 3 * 2 / sizeof (htab->entries[0]) <= n)
			enomem();
		else
			n += (n + 1) / 2;
		etab = xcalloc(n, sizeof(etab[0]));
		for (ep = hash_table_first(htab, &itr); ep; ep = hash_table_next(itr)) {
			size_t i = hash_string(ep->name, ep->namelen, n);
			struct hash_entry **epp;
			for (epp = &etab[i]; *epp; epp = &((*epp)->coll))
				;
			ep->coll = NULL;
			*epp = ep;
		}
		free(itr);
		free(htab->entries);
		htab->entries = etab;
		htab->hash_size = n;
		return hash_table_find(htab, name, namelen, 1);
	}

	head = xcalloc(1, sizeof(*head));
	head->name = xmalloc(namelen);
	memcpy(head->name, name, namelen);
	head->namelen = namelen;
	head->data = NULL;

	*tail = head;
	htab->entry_count++;

	return head;
}

void
hash_table_delete(struct hash_table *htab, const char *name, size_t namelen)
{
	size_t hv = hash_string(name, namelen, htab->hash_size);
	struct hash_entry **tail = &htab->entries[hv];
	struct hash_entry *head;

	for (;;) {
		head = *tail;
		if (head == NULL)
			return;
		if (namelen == head->namelen && memcmp(head->name, name, namelen) == 0)
			break;

		tail = &head->coll;
	}

	*tail = head->coll;
	htab->entry_count--;
	free(head->name);
	free(head);
}

static struct point syncline_point;

static void
syncline(int off)
{
	if (synclines_option) {
		syncline_point = curpoint;
		if (off)
			point_advance_line(&syncline_point, off);
	}
}

static void
syncline_print(struct point *pt)
{
	fprintf(yyout, "#line %d \"%s\"\n", pt->line, pt->file);
}

static void
syncline_flush(void)
{
	if (beginning_of_line && syncline_point.line > 0 &&
	    (YYSTATE == INITIAL || YYSTATE == ALT)) {
		syncline_print(&syncline_point);
		syncline_point.line = 0;
	}
}

extern char **environ;

/*
 * Same as getenv(3), but uses first LEN bytes of IDENT as the name of
 * a variable.
 */
static char *
findenv(char const *ident, int len)
{
	size_t i;
	for (i = 0; environ[i]; i++) {
		size_t j;
		for (j = 0; j < len; j++)
			if (environ[i][j] != ident[j])
				break;
		if (j == len && environ[i][j] == '=')
			return environ[i] + j + 1;
	}
	return NULL;
}

/*
 * Expand environment variable whose name is given by first LEN bytes of
 * IDENT.
 */
static void
expandenv(char const *ident, int len)
{
	char *val = findenv(ident, len);

	syncline_flush();

	if (val) {
		echo(val, strlen(val));
		beginning_of_line = 0;
		if (strchr(val, '\n'))
			syncline(1);
	} else {
		if (undef_error_option) {
			error_at_locus(&curlocus, "variable %.*s not defined",
				       len, ident);
			status = EX_DATAERR;
		}
		if (retain_unexpanded_option) {
			fputc('$', yyout);
			beginning_of_line = 0;
			if (env_prefix)
				echo(env_prefix, strlen(env_prefix));
			ECHO;
		}
	}
}

/*
 * Copy data from SRC to DST.  Unless BL is NULL, store there 1 if the
 * last byte copied was newline and 0 otherwise.
 */
int
fcopy(FILE *dst, FILE *src, int *bl)
{
	char buf[BUFSIZ];
	size_t n;
	int len = 0;
	while ((n = fread(buf, 1, sizeof(buf), src)) != 0) {
		fwrite(buf, 1, n, dst);
		len = n;
	}
	if (bl)
		*bl = len > 0 && buf[len-1] == '\n';
	return ferror(src);
}

static FILE *save_yyout;
static HASH_TABLE *div_table;

struct diversion {
	FILE *fp;
	struct point start_point;
};

static struct diversion *
div_get(char const *name, size_t len, int insert)
{
	HASH_ENTRY *ent;
	struct diversion *div;

	if (!div_table)
		div_table = hash_table_alloc(16);
	ent = hash_table_find(div_table, name, len, insert);
	if (!ent)
		return NULL;
	if (ent->data == NULL) {
		FILE *fp = stash_open();
		if (!fp)
			xenv_error(EX_OSERR, errno,
				   "can't create temporary file");
		div = xmalloc(sizeof(*div));
		div->fp = fp;
		div->start_point = curpoint;
		ent->data = div;
	}
	return ent->data;
}

static void
divert_start(char const *name, size_t len)
{
	struct diversion *div;

	div = div_get(name, len, 1);
	if (!save_yyout)
		save_yyout = yyout;
	yyout = div->fp;
}

static void
divert_stop(void)
{
	if (save_yyout) {
		yyout = save_yyout;
		save_yyout = NULL;
	}
}

static void
diversion_free(void *ptr)
{
	struct diversion *dp = ptr;
	fclose(dp->fp);
	free(dp);
}

static void
divert_destroy(void)
{
	hash_table_free(div_table, diversion_free);
}

static void
undivert(char const *name, size_t len)
{
	struct diversion *div = div_get(name, len, 0);
	if (!div) {
		error_at_locus(&curlocus, "no such diversion: %.*s",
			       (int)len, name);
		status = EX_DATAERR;
	} else {
		struct point save_point = curpoint;

		curpoint = div->start_point;
		syncline(1);
		rewind(div->fp);
		fcopy(yyout, div->fp, &beginning_of_line);
		curpoint = save_point;
		syncline(1);
	}
}

static void context_push(FILE *fp, struct locus *loc, dev_t dev, ino_t ino,
			 int noclose, void (*cleanup)(void*), void *data);

static void
evaldivert(char const *name, size_t len)
{
	struct diversion *div = div_get(name, len, 0);
	if (!div) {
		error_at_locus(&curlocus, "no such diversion: %.*s",
			       (int)len, name);
		status = EX_DATAERR;
	} else {
		context_push(div->fp, &curlocus, -1, -1, 1, NULL, NULL);
		curpoint = div->start_point;
		syncline(1);
		rewind(div->fp);
	}
}

static void
dropdivert(char const *name, size_t len)
{
	struct diversion *div = div_get(name, len, 0);
	if (!div) {
		error_at_locus(&curlocus, "no such diversion: %.*s",
			       (int)len, name);
		status = EX_DATAERR;
	} else {
		fclose(div->fp);
		hash_table_delete(div_table, name, len);
		free(div);
	}
}

struct saved_state {
	int state;
	struct point point;
};

static XENV_STACK state_stack = XENV_STACK_INITIALIZER(struct saved_state);

static struct point const *
start_point(void)
{
	struct saved_state *st = xenv_stack_peek(&state_stack, -1);
	return st ? &st->point : &curpoint;
}

static void
push_state_at_point(int newstate, struct point const *point)
{
	struct saved_state *st = xenv_stack_push(&state_stack);
	st->state = YYSTATE;
	st->point = *point;
	BEGIN(newstate);
}

static void
push_state(int newstate)
{
	push_state_at_point(newstate, &curpoint);
}

static void
pop_state(void)
{
	struct saved_state *st = xenv_stack_pop(&state_stack);
	BEGIN(st->state);
}

struct expand_inline {
	int type;      /* Expansion type (one of the EXP_ constants above). */
	int suppress;  /* Suppress output. */
	int synclines; /* Saved synclines_option. */
	int bracecount;/* Brace nesting level this expansion was created at. */
	int bol;       /* Beginning of line status */
	FILE *fp;      /* Previous value of yyout */
	union {
		char *varname; /* type == EXP_ASSIGN - assign the expansion
				  to that variable */
		struct locus locus; /* type == EXP_ERROR - location of the error */
	};
};

static XENV_STACK expand_inline_stack =
	XENV_STACK_INITIALIZER(struct expand_inline);

struct expand_inline *
expand_inline_peek(void)
{
	return xenv_stack_peek(&expand_inline_stack, -1);
}

/*
 * Push on stack the inline expansion entry of given TYPE.
 * VARNAME and VARLEN identify the variable for types: EXP_ASSIGN,
 * EXP_SPREF, EXP_LPREF, EXP_SSUF, EXP_LSUF, EXP_PAT, and
 * EXP_SUBST.
 * SUPPRESS is 1 if the output is to be suppressed and 0 otherwise.
 */
static void
expand_inline_push(int type, char *varname, int varlen, int suppress)
{
	struct expand_inline *ep;

	ep = xenv_stack_push(&expand_inline_stack);
	ep->type = type;
	ep->synclines = synclines_option;
	switch (type) {
	case EXP_ASSIGN:
	case EXP_SPREF:
	case EXP_LPREF:
	case EXP_SSUF:
	case EXP_LSUF:
	case EXP_PAT:
		assert(varname && varlen);
		ep->varname = xmalloc(varlen + 1);
		memcpy(ep->varname, varname, varlen);
		ep->varname[varlen] = 0;
		break;

	case EXP_ERROR:
		ep->locus = curlocus;
		break;
	}
	ep->fp = yyout;
	ep->suppress = suppress;
	ep->bracecount = bracecount;
	ep->bol = beginning_of_line;
	if (suppress) {
		if ((yyout = sink_open()) == NULL)
			xenv_error(EX_OSERR, errno, "can't open sink");
	} else if ((yyout = stash_open()) == NULL) {
		xenv_error(EX_OSERR, errno, "can't open stash");
	}
}

/*
 * Flop the suppression state of the ternary operator on the top of stack.
 * If there's no ternary on tos or its brace count doesn't match the current
 * one, echo yytext.
 */
static void
expand_inline_flop(void)
{
	struct expand_inline *ep = xenv_stack_peek(&expand_inline_stack, -1);
	if (ep &&
	    ep->bracecount == bracecount &&
	    ep->type == EXP_TERNARY) {
		int suppress = ep->suppress;
		expand_inline_pop();
		expand_inline_push(EXP_TERNARY, NULL, 0, !suppress);
	} else
		ECHO;
}

static char *
slurp(FILE *fp, size_t *ret_size)
{
	long size;
	char *value;
	size_t n;

	fseek(fp, 0, SEEK_END);
	size = ftell(fp);
	value = xmalloc(size + 1);
	rewind(fp);
	n = fread(value, 1, size, fp);
	if (n == -1) {
		xenv_error(EX_OSERR, errno, "read error");
	} else if (n > 0 && n < size) {
		xenv_error(EX_OSERR, 0, "short read");
	}
	value[n] = 0;
	if (ret_size)
		*ret_size = size;
	return value;
}

static char *
match_left(char *pattern, char *string, size_t size, size_t *ret_len)
{
	size_t len = 0;
	while (len < size) {
		if (wildmatch(pattern, string, len) == 0) {
			*ret_len = len;
			return string;
		}
		len++;
	}
	return NULL;
}

static char *
scan_left(char *pattern, char *string, size_t size, size_t *ret_len)
{
	while (1) {
		if (wildmatch(pattern, string, size) == 0) {
			*ret_len = size;
			return string;
		}
		if (size == 0)
			break;
		string++;
		size--;
	}
	return NULL;
}

static char *
match_right(char *pattern, char *string, size_t size, size_t *ret_len)
{
	while (1) {
		if (wildmatch(pattern, string, size) == 0) {
			*ret_len = size;
			return string;
		}
		if (size == 0)
			break;
		size--;
	}
	return NULL;
}

static char *
scan_right(char *pattern, char *string, size_t size, size_t *ret_len)
{
	size_t len = 0;
	char *str = string + size;
	while (len < size) {
		if (wildmatch(pattern, str, len) == 0) {
			*ret_len = len;
			return str;
		}
		str--;
		len++;
	}
	return NULL;
}

/* Strip from the value of the envvar ep->varname prefix (if pref) or suffix
   that matches the globbing pattern from yyout as per the matching function
   match. */
static void
strip(struct expand_inline *ep,
      char *(match)(char *, char *, size_t, size_t *), int pref)
{
	size_t size;
	char *pattern;
	char *value = getenv(ep->varname);

	if (value == NULL) {
		if (undef_error_option) {
			error_at_locus(&curlocus, "variable %s not defined",
				       ep->varname);
			status = EX_DATAERR;
		}
		return;
	}
	pattern = slurp(yyout, &size);
	if (!(size == 0 && value[0] == 0)) {
		size_t len = strlen(value);
		if (match(pattern, value, len, &size)) {
			if (pref)
				value += size;
			len -= size;
		}
		if (len > 0)
			fwrite(value, len, 1, ep->fp);
		ep->bol = len > 0 && value[len-1] == '\n';
	}
	free(pattern);
}

int
exp_subst_upgrade(void)
{
	struct expand_inline *ep = xenv_stack_peek(&expand_inline_stack, -1);

	if (ep->type != EXP_PAT)
		return -1;
	fputc(0, yyout);
	ep->type = EXP_SUBST;
	return 0;
}

void
exp_subst_finish(struct expand_inline *ep)
{
	char *buf, *pattern, *string;
	char *value = getenv(ep->varname);
	size_t len, size;

	if (value == NULL) {
		if (undef_error_option) {
			error_at_locus(&curlocus, "variable %s not defined",
				       ep->varname);
			status = EX_DATAERR;
		}
		return;
	}
	len = strlen(value);

	buf = slurp(yyout, 0);
	pattern = buf;
	string = buf + strlen(buf) + 1;

	switch (*pattern) {
	case '#':
		if (match_right(pattern + 1, value, len, &size)) {
			fputs(string, ep->fp);
			if (size < len)
				fputs(value + size, ep->fp);
		} else
			fputs(value, ep->fp);
		break;

	case '%':
		if (scan_left(pattern + 1, value, len, &size)) {
			if (size < len)
				fwrite(value, len-size, 1, ep->fp);
			fputs(string, ep->fp);
		} else
			fputs(value, ep->fp);
		break;

	case '/':
		/* Replace all occurrences. */
		pattern++;
		while (len > 0) {
			if (match_right(pattern, value, len, &size)) {
				fputs(string, ep->fp);
				value += size;
				len -= size;
			} else {
				fputc(*value, ep->fp);
				value++;
				len--;
			}
		}
		break;

	case '\\':
		/* Initial backslash may be used to escape special
		   characters.  The scanner rule above makes sure it
		   is retained before another backslash as well, which
		   is compensated here. */
		pattern++;
		/* fall through */
	default:
		/* Replace first occurrence. */
		while (len > 0) {
			if (match_right(pattern, value, len, &size)) {
				fputs(string, ep->fp);
				if (size < len)
					fputs(value + size, ep->fp);
				break;
			} else {
				fputc(*value, ep->fp);
				value++;
				len--;
			}
		}
		break;
	}

	free(buf);
}
/*
 * Pop the topmost expansion entry off the stack.
 * Return 1 if the operation is not applicable (wrong state or brace level
 * mismatch).
 * Return 0 on success.
 */
static int
expand_inline_pop(void)
{
	struct expand_inline *ep;

	ep = xenv_stack_peek(&expand_inline_stack, -1);
	if (ep == NULL || ep->bracecount != bracecount) {
		return 1;
	}
	xenv_stack_pop(&expand_inline_stack);
	synclines_option = ep->synclines;
	if (!ep->suppress) {
		switch (ep->type) {
		case EXP_ASSIGN: {
			size_t size;
			char *value = slurp(yyout, &size);
			setenv(ep->varname, value, 1);
			fwrite(value, size, 1, ep->fp);
			if (strchr(value, '\n'))
				syncline(1);
			free(value);
			break;
		}

		case EXP_SPREF: /* $(VAR#PAT} */
			strip(ep, match_left, 1);
			break;

		case EXP_LPREF: /* $(VAR##PAT} */
			strip(ep, match_right, 1);
			break;

		case EXP_SSUF: /* $(VAR%PAT} */
			strip(ep, scan_right, 0);
			break;

		case EXP_LSUF: /* $(VAR%%PAT} */
			strip(ep, scan_left, 0);
			break;

		case EXP_PAT:
			fputc(0, yyout);
		case EXP_SUBST:
			exp_subst_finish(ep);
			break;

		case EXP_ERROR:
			ep->locus.end = curpoint;
			locus_print(stderr, &ep->locus);
			fprintf(stderr, ": ");
			rewind(yyout);
			fcopy(stderr, yyout, NULL);
			fputc('\n', stderr);
			break;

		default:
			rewind(yyout);
			fcopy(ep->fp, yyout, &ep->bol);
			syncline(1);
		}
	} else
		/* FIXME: Emit syncline only if there were newlines in the
		   suppressed text. */
		syncline(1);

	switch (ep->type) {
	case EXP_ASSIGN:
	case EXP_SPREF:
	case EXP_LPREF:
	case EXP_SSUF:
	case EXP_LSUF:
	case EXP_PAT:
	case EXP_SUBST:
		free(ep->varname);
		break;
	}

	fclose(yyout);
	yyout = ep->fp;
	beginning_of_line = ep->bol;
	return 0;
}

/*
 * Condition stack.  Implements conditional preprocessor blocks like:
 *
 *   $$ifset
 *    TEXT1
 *   $$else
 *    TEXT2
 *   $$endif
 */

struct cond_state {
	int result;
	struct point point;
};

static XENV_STACK cond_stack = XENV_STACK_INITIALIZER(struct cond_state);

/*
 * Return the evaluation result from the top of stack.
 */
static int
cond_peek(struct point *point)
{
	struct cond_state *st = xenv_stack_peek(&cond_stack, -1);
	if (st == NULL)
		return -1;
	if (point)
		*point = st->point;
	return st->result;
}

/*
 * Push RESULT on stack. Switch to state FALSE if result is false. Otherwise,
 * output a syncline.
 */
static void
cond_push(int result)
{
	struct cond_state *st = xenv_stack_push(&cond_stack);
	st->result = result;
	st->point = curlocus.beg;
	if (!result)
		push_state_at_point(FALSE, &curlocus.beg);
	else
		syncline(1);
}

/*
 * Pop the entry off the stack.  If it is false, restore previous scanner
 * state.  Return the evaluation result.
 */
static int
cond_pop(void)
{
	struct cond_state *st = xenv_stack_pop(&cond_stack);
	if (!st->result)
		pop_state();
	return st->result;
}

static int
parse_line_directive(char *text)
{
	unsigned long n;
	char *fname, *p;

	text += sizeof("#line ")-1;

	n = strtoul(text, &p, 10);
	if ((n == 0 && errno == ERANGE) || n > INT_MAX ||
	    !(*p == 0 || *p == ' ' || *p == '\t' || *p == '\n'))
		return 1;
	while (*p == ' ' || *p == '\t')
		p++;
	if (*p == '\n' || *p == 0) {
		curpoint.line = n;
		curpoint.column = 0;
		return 0;
	}
	if (*p != '"')
		return 1;
	p++;
	fname = p;
	p = strchr(fname, '"');
	if (!p)
		return 1;
	point_init(&curpoint, fname, p - fname);
	curpoint.line = n;
	curpoint.column = 0;
	curlocus.beg = curlocus.end = curpoint;
	return 0;
}

/*
 * Input file handling.
 */
static char **input_files; /* Array of input files. */
static int input_index;    /* Index of the next file in input_files. */

static int
file_error_exit(int ec)
{
	switch (ec) {
	case ENOENT:
		exit(EX_NOINPUT);

	case EACCES:
		exit(EX_NOPERM);

	default:
		exit(EX_OSERR);
	}
}

static void
open_input(char const *name)
{
	FILE *fp;
	struct stat st;

	if (name[0] == '-' && name[1] == 0) {
		name = "<stdin>";
		fp = stdin;
	} else {
		if ((fp = fopen(name, "r")) == NULL) {
			int ec = errno;
			xenv_error(0, ec, "can't open %s", name);
			file_error_exit(ec);
		}
	}

	yyin = fp;
	point_init(&curpoint, name, strlen(name));
	beginning_of_line = 1;
	syncline(0);

	if (fstat(fileno(yyin), &st)) {
		int ec = errno;
		xenv_error(0, ec, "can't stat file %s", name);
		file_error_exit(ec);
	}
	device = st.st_dev;
	inode = st.st_ino;
}

struct search_path_dir {
	struct search_path_dir *next;
	size_t len;
	char name[1];
};

struct search_path_dir *search_path_head, *search_path_tail;

static void
search_path_append(char *dir)
{
	size_t namelen = strlen(dir);
	struct search_path_dir *dp = malloc(sizeof(*dp) + namelen);
	dp->len = namelen;
	strcpy(dp->name, dir);
	dp->next = NULL;
	if (search_path_tail)
		search_path_tail->next = dp;
	else
		search_path_head = dp;
	search_path_tail = dp;
}

static char *
find_file(char const *name)
{
	if (!search_path_head) {
		if (access(name, F_OK) == 0)
			return xstrdup(name);
	} else {
		struct search_path_dir *dp;
		size_t namelen = strlen(name);

		char *filebuf = NULL;
		size_t filebuflen = 0;

		for (dp = search_path_head; dp; dp = dp->next) {
			size_t len = dp->len + namelen + 2;
			if (len > filebuflen) {
				filebuflen = len;
				filebuf = xrealloc(filebuf, filebuflen);
			}
			memcpy(filebuf, dp->name, dp->len);
			filebuf[dp->len] = '/';
			strcpy(filebuf + dp->len + 1, name);

			if (access(filebuf, F_OK) == 0)
				return filebuf;
			else if (errno != ENOENT)
				error_at_locus(&curlocus, "can't stat %s: %s",
					       filebuf, strerror(errno));
		}
		free(filebuf);
	}
	return NULL;
}

struct context_stack {
	YY_BUFFER_STATE state;
	char *filename;
	struct locus locus;
	ino_t inode;
	dev_t device;
	int noclose;
	void (*cleanup_fun)(void *);
	void *cleanup_data;
	struct context_stack *prev;
};

static struct context_stack *context_stack_top;

static struct context_stack *
context_find(dev_t device, ino_t inode)
{
	struct context_stack *ctx = context_stack_top;
	while (ctx) {
		if (ctx->inode == inode && ctx->device == device)
			break;
		ctx = ctx->prev;
	}
	return ctx;
}

static void
context_push(FILE *fp, struct locus *loc, dev_t dev, ino_t ino, int noclose,
	     void (*cleanup)(void *), void *data)
{
	struct context_stack *sp;

	sp = xcalloc(1, sizeof(*sp));
	sp->state = YY_CURRENT_BUFFER;
	sp->locus = *loc;
	sp->device = device;
	sp->inode = inode;
	sp->noclose = noclose;
	sp->cleanup_fun = cleanup;
	sp->cleanup_data = data;
	sp->prev = context_stack_top;
	context_stack_top = sp;

	yyin = fp;
	yy_switch_to_buffer(yy_create_buffer(yyin, YY_BUF_SIZE));
	beginning_of_line = 1;
	device = dev;
	inode = ino;
}

static void
push_input(char const *name, int oknotfound, struct locus *loc)
{
	struct context_stack *sp;
	FILE *fp;
	char *nameptr = NULL;
	struct stat st;
	int found;

	if (!(name[0] == '/' ||
	      (name[0] == '.' && name[1] == '/') ||
	      (name[0] == '.' && name[1] == '.' && name[2] == '/'))) {
		nameptr = find_file(name);
		if (!nameptr) {
			if (oknotfound) {
				return;
			}
			error_at_locus(loc, "file not found: %s", name);
			exit(EX_NOINPUT);
		}
		name = nameptr;
	}

	if (stat(name, &st)) {
		int ec;

		if (oknotfound && errno == ENOENT) {
			free(nameptr);
			return;
		}
		ec = errno;
		error_at_locus(loc, "can't stat file %s: %s",
			       name, strerror(ec));
		file_error_exit(ec);
	}

	if (st.st_ino == inode && st.st_dev == device) {
		found = 1;
		sp = context_stack_top;
	} else if ((sp = context_find(st.st_dev, st.st_ino)) != NULL) {
		sp = sp->prev;
		found = 1;
	} else
		found = 0;

	if (found) {
		error_at_locus(loc, "file %s already included", name);
		if (sp)
			error_at_locus(&sp->locus,
				       "place of the initial inclusion");
		exit(EX_UNAVAILABLE);
	}

	fp = fopen(name, "r");
	if (!fp) {
		int ec;

		if (oknotfound && errno == ENOENT) {
			free(nameptr);
			return;
		}
		ec = errno;
		error_at_locus(loc, "can't open file %s: %s",
			       name, strerror(ec));
		file_error_exit(ec);
	}

	context_push(fp, loc, st.st_dev, st.st_ino, 0, NULL, NULL);

	point_init(&curpoint, name, strlen(name));

	free(nameptr);
}

static int
pop_input(void)
{
	struct context_stack *sp = context_stack_top;

	if (yyin != stdin && !(sp && sp->noclose))
		fclose(yyin);
	yy_delete_buffer(YY_CURRENT_BUFFER);

	if (!sp)
		return 1;
	if (sp->cleanup_fun)
		sp->cleanup_fun(sp->cleanup_data);

	context_stack_top = sp->prev;

	yy_switch_to_buffer(sp->state);
	curpoint = sp->locus.end;
	point_advance_line(&curpoint, 1);
	beginning_of_line = 1;
	syncline(0);

	device = sp->device;
	inode = sp->inode;

	free(sp->filename);
	free(sp);

	return 0;
}

/* Macros */
struct macro {
	struct point start_point;
	FILE *bodyfile;
	size_t paramc;
	char **paramv;
};

static HASH_TABLE *macro_table;

static MACRO *
macro_get(char const *name, int insert)
{
	HASH_ENTRY *ent;

	if (!macro_table)
		macro_table = hash_table_alloc(16);
	ent = hash_table_find(macro_table, name, strlen(name), insert);
	if (!ent)
		return NULL;
	if (ent->data == NULL)
		ent->data = xcalloc(1, sizeof(MACRO));
	return ent->data;
}

static void
argv_free(size_t argc, char **argv)
{
	if (argv) {
		size_t i;
		for (i = 0; i < argc; i++)
			free(argv[i]);
		free(argv);
	}
}

static void
macro_free(void *ptr)
{
	MACRO *macro = ptr;
	fclose(macro->bodyfile);
	argv_free(macro->paramc, macro->paramv);
	free(macro);
}

static void
macros_destroy(void)
{
	hash_table_free(macro_table, macro_free);
}


struct argstream {
	FILE *file;            /* Argument stream. */
	struct stringbuf buf;  /* Argument buffer */
	struct locus locus;    /* Locus used when parsing arguments */
};

struct paramctx {
	enum paramctx_type type;  /* Context type */

	size_t start_line;        /* Line number where the controlling
				     statement ($$set, $$loop, $$range, etc)
				     appeared. */

	char *prefix;             /* Command prefix */

	/* The following depends on context type: */
	union {
		struct {
			char *name;   /* Name of the controlling variable */
			char *saved_value;  /* Its original value */
			struct argstream args;
			FILE *bodyfile;     /* Loop body. */
			int complete; /* $$end seen */
		} loop;

		struct {              /* PARAMCTX_RANGE */
			char *name;   /* Name of the controlling variable */
			char *saved_value;  /* Its original value */
			struct argstream args;
			FILE *bodyfile;     /* Loop body. */
			int complete; /* $$end seen */

			long cur;     /* Current value of the counter */
			long last;    /* Limiting value */
			long incr;    /* Counter increment */
		} range;

		struct {              /* PARAMCTX_FOREACH */
			char *name;   /* Name of the controlling variable */
			char *saved_value;  /* Its original value */
			struct argstream args;
			FILE *bodyfile;     /* Loop body. */
			int complete; /* $$end seen */
		} foreach;

		struct {                    /* PARAMCTX_SET */
			char *name;         /* Variable name. */
			struct argstream args;
		} setvar;

		struct {                    /* PARAMCTX_EVAL */
			FILE *bodyfile;     /* Context body. */
			struct locus locus;
			int pass;           /* Pass count */
		} eval;

		struct {
			char *name;
			struct argstream args;
			FILE *bodyfile;     /* Context body. */
			struct point start_point;
		} macro;

		struct {
			FILE *out;
		} recovery;
	};
};

static void
argstream_close(struct argstream *args)
{
	stringbuf_free(&args->buf);
	if (args->file)
	    fclose(args->file);
}

/*
 * Collect next argument from pctx->argfile.  The argument is any non-empty
 * contiguous sequence of non-whitespace characters.  Initial whitespace is
 * ignored.
 * Returns 0 on success (pctx->argbuf contains the argument and pctx->arglen
 * its actual length).
 * On EOF or error returns 1.
 */
int
argstream_getarg(struct argstream *args)
{
	int c, dq = 0;

	/* Skip whitespace */
	do {
		if ((c = fgetc(args->file)) == EOF) {
			return 1;
		}
		args->locus.end.column++;
	} while (c_isspace(c));

	/* Collect argument */
	args->locus.beg = args->locus.end;

	stringbuf_reset(&args->buf);
	for (; c != EOF;
	     c = fgetc(args->file), args->locus.end.column++) {
		if (c == '"' || c == '\'') {
			if (dq == 0) {
				dq = c;
				continue;
			} else if (dq == c) {
				dq = 0;
				continue;
			}
		} else if (dq == '"' && c == '\\') {
			c = fgetc(args->file);
			if (c == EOF) {
				stringbuf_add(&args->buf, '\\');
				break;
			}
		} else if (dq == 0 && c_isspace(c))
			break;
		stringbuf_add(&args->buf, c);
	}
	stringbuf_add(&args->buf, 0);
	return 0;
}

/*
 * Read next numeric argument from pctx->argfile,
 * Return value:
 *   0   -   Success.  Argument is stored in RET.
 *   1   -   EOF.
 *  -1   -   Argument read, but it is not numeric.
 */
int
argstream_getarg_n(struct argstream *args, long *ret)
{
	long n;
	char *p;

	if (argstream_getarg(args))
		return 1;
	errno = 0;
	n = strtol(args->buf.text, &p, 10);
	if (errno || *p)
		return -1;
	*ret = n;
	return 0;
}

static XENV_STACK paramctx_stack = XENV_STACK_INITIALIZER(struct paramctx);

/* Redirect yyout to a newly created stash.  Store its previous value in
   *prev_out. */
static inline void
paramctx_stash_setup(struct paramctx *pctx, FILE **prev_out)
{
	*prev_out = yyout;
	if ((yyout = stash_open()) == NULL)
		xenv_error(EX_OSERR, errno, "can't create temporary file");
}

static void
paramctx_eval_init(struct paramctx *pctx)
{
	paramctx_stash_setup(pctx, &pctx->eval.bodyfile);
	pctx->eval.locus.beg = curlocus.beg;
	pctx->eval.locus.end = curpoint;
	paramctx_collect();
}

static void
paramctx_loop_init(struct paramctx *pctx, char const *name, size_t nlen)
{
	char *s;
	pctx->loop.name = xmalloc(nlen+1);
	memcpy(pctx->loop.name, name, nlen);
	pctx->loop.name[nlen] = 0;
	s = getenv(pctx->loop.name);
	if (s)
		pctx->loop.saved_value = xstrdup(s);
	else
		pctx->loop.saved_value = NULL;
	paramctx_stash_setup(pctx, &pctx->loop.bodyfile);
	pctx->loop.args.locus.beg = pctx->loop.args.locus.end = curpoint;
}

static void
paramctx_set_init(struct paramctx *pctx, char const *name, size_t nlen)
{
	pctx->setvar.args.file = yyout;
	pctx->setvar.name = xmalloc(nlen + 1);
	memcpy(pctx->setvar.name, name, nlen);
	pctx->setvar.name[nlen] = 0;
	pctx->setvar.args.locus.beg = pctx->setvar.args.locus.end = curpoint;
	if ((yyout = stash_open()) == NULL)
		xenv_error(EX_OSERR, errno, "can't create temporary file");
}

static void
paramctx_macro_init(struct paramctx *pctx, char const *name, size_t nlen)
{
	pctx->macro.name = xmalloc(nlen);
	memcpy(pctx->macro.name, name, nlen);
	pctx->macro.name[nlen] = 0;
	paramctx_stash_setup(pctx, &pctx->macro.args.file);
	pctx->macro.args.locus.beg = pctx->macro.args.locus.end = curpoint;
}

static void
paramctx_recovery_init(struct paramctx *pctx)
{
	if (feature_is_set(FEATURE_RPE)) {
		pctx->recovery.out = NULL;
	} else {
		pctx->recovery.out = yyout;
		yyout = sink_open();
	}
}
/*
 * paramctx_init pushes on stack a new parametrized context of given TYPE with
 * the controlling variable NAME.  This function is called right after
 * the loop statement and its control variable name have been scanned.
 * It stashes away the current yyout in loop->argfile and reinitializes it
 * to a freshly opened temporary file.
 * Rest of fields in the loop state structure will be filled in by subsequent
 * calls to paramctx_collect and loop_begin,
 */
void
paramctx_init(int type,
	      char const *name, size_t nlen,
	      char const *prefix, size_t plen)
{
	struct paramctx *pctx = xenv_stack_push(&paramctx_stack);

	memset(pctx, 0, sizeof *pctx);
	pctx->type = type;

	pctx->prefix = xmalloc(plen+1);
	if (plen > 0)
		memcpy(pctx->prefix, prefix, plen);
	pctx->prefix[plen] = 0;

	switch (type) {
	case PARAMCTX_RECOVERY:
		paramctx_recovery_init(pctx);
		break;

	case PARAMCTX_FOREACH:
	case PARAMCTX_RANGE:
		paramctx_loop_init(pctx, name, nlen);
		break;

	case PARAMCTX_EVAL:
		paramctx_eval_init(pctx);
		break;

	case PARAMCTX_SET:
		paramctx_set_init(pctx, name, nlen);
		break;

	case PARAMCTX_DEFMACRO:
	case PARAMCTX_CALLMACRO:
		paramctx_macro_init(pctx, name, nlen);
		break;

	default:
		abort();
	}
}

static void
paramctx_restorevar(struct paramctx *ctx)
{
	if (ctx->loop.saved_value) {
		setenv(ctx->loop.name, ctx->loop.saved_value, 1);
		free(ctx->loop.saved_value);
	} else
		unsetenv(ctx->loop.name);
}

/*
 * Pop the topmost paramctx off the stack and close it, freeing any
 * associated resources and restoring initial value of the controlling
 * variable.
 */
void
paramctx_close(void)
{
	struct paramctx *pctx = xenv_stack_pop(&paramctx_stack);

	switch (pctx->type) {
	case PARAMCTX_RECOVERY:
		if (pctx->recovery.out) {
			fclose(yyout);
			yyout = pctx->recovery.out;
		}
		break;

	case PARAMCTX_FOREACH:
	case PARAMCTX_RANGE:
		paramctx_restorevar(pctx);
		free(pctx->loop.name);
		argstream_close(&pctx->loop.args);
		/* pctx->loop.bodyfile is closed by pop_input */
		break;

	case PARAMCTX_EVAL:
		break;

	case PARAMCTX_SET:
		free(pctx->setvar.name);
		argstream_close(&pctx->setvar.args);
		break;

	case PARAMCTX_DEFMACRO:
	case PARAMCTX_CALLMACRO:
		free(pctx->macro.name);
		argstream_close(&pctx->macro.args);
		break;

	default:
		abort();
	}
	free(pctx->prefix);
}

/*
 * Rudimentary recovery from syntax errors occurring in PARAMS state.
 * Prints out the original construct with the arguments expanded.
 */
void
paramctx_recover_args(struct paramctx *pctx)
{
	FILE *fp;

	if (!feature_is_set(FEATURE_RPE))
		return;

	if (!pctx)
		pctx = xenv_stack_peek(&paramctx_stack, -1);

	switch (pctx->type) {
	case PARAMCTX_FOREACH:
	case PARAMCTX_RANGE:
		fp = pctx->loop.args.file;
		break;

	case PARAMCTX_EVAL:
		fp = NULL;
		break;

	case PARAMCTX_SET:
		fp = pctx->setvar.args.file;
		break;

	case PARAMCTX_DEFMACRO:
	case PARAMCTX_CALLMACRO:
		fp = pctx->macro.args.file;
		break;

	default:
		abort();
	}
	fputs("$$", yyout);
	fputs(pctx->prefix, yyout);
	rewind(fp);
	fcopy(yyout, fp, NULL);
	fputc('\n', yyout);
}

static inline void
swapfiles(FILE **a, FILE **b)
{
	FILE *tmp = *a;
	*a = *b;
	*b = tmp;
}

static void
paramctx_recover(void)
{
	struct paramctx *pctx = xenv_stack_peek(&paramctx_stack, -1);
	if (!pctx)
		return;
	switch (pctx->type) {
	case PARAMCTX_RECOVERY:
		break;

	case PARAMCTX_FOREACH:
	case PARAMCTX_RANGE:
		if (!pctx->loop.complete)
			swapfiles(&yyout, &pctx->loop.bodyfile);
		paramctx_recover_args(pctx);
		fcopy(yyout, pctx->loop.bodyfile, NULL);
		fclose(pctx->loop.bodyfile);
		pctx->loop.bodyfile = NULL;
		break;
	default:
		paramctx_recover_args(pctx);
	}
}

int
paramctx_type(void)
{
	struct paramctx *loop = xenv_stack_peek(&paramctx_stack, -1);
	return (loop != NULL) ? loop->type : PARAMCTX_NONE;
}

static void
paramctx_setvar(void)
{
	struct paramctx *ctx = xenv_stack_peek(&paramctx_stack, -1);
	char const *val;
	char *valbuf = NULL;

	swapfiles(&yyout, &ctx->setvar.args.file);

	if (argstream_getarg(&ctx->setvar.args)) {
		val = "";
	} else {
		val = valbuf = ctx->setvar.args.buf.text;

		stringbuf_init(&ctx->setvar.args.buf);
		if (argstream_getarg(&ctx->setvar.args) == 0) {
			error_at_locus(&ctx->setvar.args.locus,
				       "extra arguments");
			status = EX_DATAERR;
			paramctx_recover_args(ctx);
			free(valbuf);
			return;
		}
	}
	setenv(ctx->setvar.name, val, 1);
	free(valbuf);
}

/*
 * Prepare loop state for iterating over a range of values.  To do so,
 * the function scans loop->argfile for a sequence of whitespace delimited
 * numeric arguments.  First argument sets the initial value of the control
 * variable, second argument sets the last value, and optional third value
 * sets the increment.  If not given, the increment is +1 if limit is greater
 * than the start value and -1 otherwise.
 *
 * Returns 0 on success and -1 on failure.  The caller should save the value
 * of loop->argfile before calling loop_range and fclose it afterwards, no
 * matter what the returned value.
 */
int
loop_range(struct paramctx *loop)
{
	long start, stop, incr;

	rewind(loop->range.args.file);
	switch (argstream_getarg_n(&loop->range.args, &start)) {
	case 0:
		break;
	case 1:
		error_at_point(&loop->range.args.locus.end,
			       "unexpected end of line");
		return -1;
	case -1:
		error_at_locus(&loop->range.args.locus, "not a number");
		return -1;
	}

	switch (argstream_getarg_n(&loop->range.args, &stop)) {
	case 0:
		break;
	case 1:
		error_at_point(&loop->range.args.locus.end,
			       "unexpected end of line");
		return -1;
	case -1:
		error_at_locus(&loop->range.args.locus, "not a number");
		return -1;
	}

	switch (argstream_getarg_n(&loop->range.args, &incr)) {
	case 0:
		if (incr == 0 ||
		    (stop > start && incr < 0) || (stop < start && incr > 0)) {
			error_at_locus(&loop->range.args.locus,
				       "invalid increment value");
		}
		break;
	case 1:
		incr = stop > start ? 1 : -1;
		break;
	case -1:
		error_at_locus(&loop->range.args.locus, "not a number");
		return -1;
	}

	if (argstream_getarg(&loop->range.args) == 0) {
		error_at_locus(&loop->range.args.locus,
			       "unexpected extra argument");
		return -1;
	}

	loop->range.cur = start;
	loop->range.last = stop;
	loop->range.incr = incr;

	return 0;
}

/*
 * Collect loop body into temporary file loop->bodyfile.  On entry,
 * loop->bodyfile keeps the stashed output stream, and yyout points to a
 * temporary file with collected loop arguments.
 *
 * The function assigns yyout to loop->argfile. and reinitializes it to
 * a freshly opened temporary file where the loop body will be copied to.
 * This stream will be moved to loop->bodyfile by loop_begin.
 *
 * This function is called when a newline after loop arguments is seen.
 */
void
paramctx_collect(void)
{
	struct paramctx *pctx = xenv_stack_peek(&paramctx_stack, -1);

	pctx->start_line = curpoint.line;
	switch (pctx->type) {
	case PARAMCTX_FOREACH:
		pctx->foreach.args.file = yyout;
		break;
	case PARAMCTX_RANGE:
		pctx->range.args.file = yyout;
		if (loop_range(pctx)) {
			status = EX_DATAERR;

			yyout = pctx->range.bodyfile;
			paramctx_recover_args(pctx);
			paramctx_close();
			pop_state();

			paramctx_init(PARAMCTX_RECOVERY, NULL, 0, NULL, 0);
			push_state(COLLECT);
			//FIXME
			syncline(0);
			beginning_of_line = 1;
			return;
		}
		break;
	case PARAMCTX_EVAL:
		pctx->eval.pass = 0;
		return;

	case PARAMCTX_DEFMACRO:
		pctx->macro.start_point = curpoint;
		swapfiles(&yyout, &pctx->macro.args.file);
		pctx->macro.bodyfile = yyout;
/* FIXME: do
 *	  paramctx_stash_setup(pctx, &pctx->macro.bodyfile);
 * instead of assignment to yyout below?
 */
		break;

	default:
		abort();
	}

	if ((yyout = stash_open()) == NULL)
		xenv_error(EX_OSERR, errno, "can't create temporary file");
}

static int
is_varname(char const *s)
{
	if (!(c_isalpha(*s) || *s == '_'))
		return 0;
	while (*++s) {
		if (!(c_isalnum(*s) || *s == '_'))
			return 0;
	}
	return 1;
}

static void
paramctx_register_macro(struct paramctx *pctx)
{
	size_t paramc = 0, paramn = 0;
	char **paramv = NULL;
	MACRO *macro;

	rewind(pctx->macro.args.file);
	while (argstream_getarg(&pctx->macro.args) == 0) {
		if (!is_varname(pctx->macro.args.buf.text)) {
			error_at_locus(&pctx->macro.args.locus,
				       "not a valid variable name");
			status = EX_DATAERR;
			argv_free(paramc, paramv);
			fclose(pctx->macro.bodyfile);
			return;
		}
		if (paramc == paramn)
			paramv = x2nrealloc(paramv, &paramn, sizeof(*paramv));
		paramv[paramc++] = xstrdup(pctx->macro.args.buf.text);
		stringbuf_reset(&pctx->macro.args.buf);
	}

	macro = macro_get(pctx->macro.name, 1);
	if (macro->bodyfile) {
		error_at_point(&pctx->macro.start_point,
			       "redefinition of macro %s", pctx->macro.name);
		error_at_point(&macro->start_point,
			       "this is the place of previous definition");
		status = EX_DATAERR;
		argv_free(paramc, paramv);
		fclose(pctx->macro.bodyfile);
		return;
	}
	macro->start_point = pctx->macro.start_point;
	macro->bodyfile = pctx->macro.bodyfile;
	macro->paramc = paramc;
	macro->paramv = paramv;
}

struct call_context {
	MACRO *macro;
	char **value;
};

static void
call_cleanup(void *data)
{
	struct call_context *ctx = data;
	size_t i;
	for (i = 0; i < ctx->macro->paramc; i++) {
		if (ctx->value[i]) {
			setenv(ctx->macro->paramv[i], ctx->value[i], 1);
			free(ctx->value[i]);
		} else
			unsetenv(ctx->macro->paramv[i]);
	}
	free(ctx->value);
	free(ctx);
}


static void
paramctx_callmacro(void)
{
	struct paramctx *pctx = xenv_stack_peek(&paramctx_stack, -1);
	MACRO *macro;
	struct call_context *cctx;
	size_t i;

	swapfiles(&yyout, &pctx->macro.args.file);

	macro = macro_get(pctx->macro.name, 0);
	if (!macro) {
		// FIXME: error location
		error_at_point(&curpoint, "macro not defined");
		status = EX_DATAERR;
		paramctx_recover_args(pctx);
		return;
	}

	/* Prepare call context */
	cctx = xcalloc(1, sizeof(*cctx));
	cctx->macro = macro;
	cctx->value = xcalloc(macro->paramc, sizeof(cctx->value[0]));
	for (i = 0; i < macro->paramc; i++) {
		char *p = getenv(macro->paramv[i]);
		cctx->value[i] = p ? xstrdup(p) : NULL;
	}

	rewind(pctx->macro.args.file);
	for (i = 0;
	     i < macro->paramc && argstream_getarg(&pctx->macro.args) == 0;
	     i++)
		setenv(macro->paramv[i], pctx->macro.args.buf.text, 1);
	for (; i < macro->paramc; i++)
		unsetenv(macro->paramv[i]);
	if (argstream_getarg(&pctx->macro.args) == 0)
		error_at_locus(&pctx->macro.args.locus,
			       "extra arguments to %s ignored",
			       pctx->macro.name);
	rewind(macro->bodyfile);
	context_push(macro->bodyfile, &curlocus, -1, -1, 1, call_cleanup,
		     cctx);
	curpoint = macro->start_point;
	syncline(1);
}

/*
 * Begin loop iterations.  This function is called after the terminating
 * $$end is scanned.  On entry, loop->bodyfile keeps the stashed away
 * output stream and yyout points to a temporary file with the scanned
 * loop body.  The function will swap the two, and push a new input
 * context using loop->bodyfile as input.  The context will further be
 * maintained by calls to loop_next().
 */
void
loop_begin(void)
{
	struct paramctx *pctx = xenv_stack_peek(&paramctx_stack, -1);
	FILE *input;

	switch (pctx->type) {
	case PARAMCTX_RECOVERY:
		ECHO;
		paramctx_close();
		beginning_of_line = 1;
		syncline(1);
		return;

	case PARAMCTX_FOREACH:
		pctx->foreach.complete = 1;
		swapfiles(&yyout, &pctx->foreach.bodyfile);
		rewind(pctx->foreach.args.file);
		input = pctx->foreach.bodyfile;
		break;

	case PARAMCTX_RANGE:
		pctx->foreach.complete = 1;
		swapfiles(&yyout, &pctx->range.bodyfile);
		input = pctx->range.bodyfile;
		break;

	case PARAMCTX_EVAL:
		swapfiles(&yyout, &pctx->eval.bodyfile);
		pctx->eval.pass++;
		input = pctx->eval.bodyfile;
		break;

	case PARAMCTX_DEFMACRO:
		swapfiles(&yyout, &pctx->macro.bodyfile);
		paramctx_register_macro(pctx);
		paramctx_close();
		beginning_of_line = 1;
		syncline(1);
		return;

	default:
		abort();
	}

	context_push(input, &curlocus, -1, -1, 0, NULL, NULL);
	loop_next();
	curpoint.line--;
}

/*
 * Start next loop iteration.
 *
 * Returns 0 on success, and 1 to break off the loop.  In the latter case,
 * calls paramctx_close() before returning.
 */
int
loop_next(void)
{
	struct paramctx *pctx;

	if ((pctx = xenv_stack_peek(&paramctx_stack, -1)) == NULL)
		return 1;

	switch (pctx->type) {
	case PARAMCTX_FOREACH:
		if (!pctx->foreach.complete)
			return 1;
		if (argstream_getarg(&pctx->foreach.args)) {
			paramctx_close();
			return 1;
		}
		setenv(pctx->foreach.name, pctx->foreach.args.buf.text, 1);
		break;

	case PARAMCTX_RANGE:
		if (!pctx->range.complete)
			return 1;
		else {
			long n;
			int sign;

			if ((pctx->range.incr > 0 &&
			     pctx->range.cur > pctx->range.last) ||
			    (pctx->range.incr < 0 &&
			     pctx->range.cur < pctx->range.last)) {
				paramctx_close();
				return 1;
			}

			n = pctx->range.cur;
			pctx->range.cur += pctx->range.incr;

			if (n < 0) {
				n = - n;
				sign = 1;
			} else
				sign = 0;

			stringbuf_reset(&pctx->range.args.buf);
			do {
				long x = n % 10;
				stringbuf_add(&pctx->range.args.buf, x + '0');
				n /= 10;
			} while (n > 0);
			if (sign == 1)
				stringbuf_add(&pctx->range.args.buf, '-');
			stringbuf_revert(&pctx->range.args.buf);
			stringbuf_add(&pctx->range.args.buf, 0);
			setenv(pctx->range.name, pctx->range.args.buf.text, 1);
		}
		break;

	case PARAMCTX_EVAL:
		switch (pctx->eval.pass++) {
		case 0:
			error_at_point(&pctx->eval.locus.beg,
				       "end of file in eval");
			status = EX_DATAERR;
			fcopy(pctx->eval.bodyfile, yyout, NULL);
			/* fall through */
		case 2:
			paramctx_close();
			return 1;
		case 1:
			break;
		}
		break;

	default:
		return 1;
	}

	/* Push input */
	rewind(yyin);

	curpoint.line = pctx->start_line;
	curpoint.column = 0;
	beginning_of_line = 1;
	syncline(0);

	return 0;
}

int
yywrap(void)
{
	if (loop_next() == 0)
		return 0;
	if (pop_input() == 0)
		return 0;
	if (input_files[input_index]) {
		open_input(input_files[input_index++]);
		return 0;
	}
	return 1;
}

/*
 * Process input from the current position up to and including the closing
 * delimiter (CD).  It is supposed that the function is called immediately
 * after the opening delimiter (OD) was seen.  For each consumed character,
 * call COLLECTOR with CLOSURE as the first argument and the character read
 * as the second.  Assume each OD is paired by CD.
 */
static int
collect_delimited_text(int od, int cd, void (*collector)(void *, int),
		       void *closure)
{
	int nesting = 1;
	int c;
	/*
	 * On end of file, input() returns EOF in flex versions
	 * before 2.6.1, and 0 in flex 2.6.1 and later.
	 * See https://github.com/westes/flex/issues/448
	 */
	while ((c = input()) != EOF && c != 0) {
		curpoint.column++;
		if (c == od)
			nesting++;
		else if (c == cd) {
			if (--nesting == 0)
				break;
		} else if (c == '\n')
			point_advance_line(&curpoint, 1);
		collector(closure, c);
	}
	curlocus.end = curpoint;
	return c == 0 ? EOF : c;
}

static void
ignore_text(void)
{
	char const buf[] = "$$end\n";
	int i = 0;
	int c;
	struct point p = curpoint;

	while ((c = input()) != EOF && c != 0) {
		if (c == '\n')
			point_advance_line(&curpoint, 1);

		if (i != -1) {
			if ((i == 0 || i == 2 || i == 5) && c_isblank(c))
				continue;
			if (c == buf[i]) {
				if (buf[++i] == 0)
					break;
				continue;
			}
				i = -1;
		}
		i = (c == '\n') ? 0 : -1;
	}
	if (i == -1 || buf[i]) {
		error_at_point(&p, "end of file in ignore block");
		exit(EX_DATAERR);
	} else if (synclines_option)
		syncline_print(&curpoint);
}

static inline int
timeval_cmp(struct timeval const *a, struct timeval const *b)
{
	if (a->tv_sec < b->tv_sec)
		return -1;
	if (a->tv_sec > b->tv_sec)
		return 1;
	if (a->tv_usec < b->tv_usec)
		return -1;
	if (a->tv_usec > b->tv_usec)
		return 1;
	return 0;
}

static inline int
timeval_diff(struct timeval const *a, struct timeval const *b)
{
	struct timeval d;

	d.tv_sec = a->tv_sec - b->tv_sec;
	d.tv_usec = a->tv_usec - b->tv_usec;
	if (d.tv_usec < 0) {
		--d.tv_sec;
		d.tv_usec += 1000000;
	}
	return d.tv_sec * 1000 + d.tv_usec / 1000;
}

static int
countnl(char const *s, size_t l)
{
	int n = 0;
	for (; l; s++, l--)
		if (*s == '\n')
			n++;
	return n;
}

enum {
	S_OUT,
	S_ERR,
	S_NUM
};

typedef void (*runcom_capture_fn) (void *data, int dir, char *str, size_t len,
				   struct locus const *locus);

/*
 * Run command line from first LEN bytes of STR, capturing its standard
 * output and error streams.  For each chunk of text obtained from these,
 * invoke FN with dir set to S_OUT or S_ERR, str and len containing the
 * obtained chunk, data and locus pointing to corresponding arguments
 * of runcom.  Before returning, FN is additionally called with str=NULL
 * and LEN=0 for eventual end of input actions.
 *
 * If command_timeout is set, make sure the command is terminated if
 * it runs longer than that.
 */
static int
runcom(char *str, size_t len, runcom_capture_fn fn, void *data,
       struct locus const *loc)
{
	pid_t pid;
	int p[2][2];

	syncline_flush();

	if (pipe(p[S_OUT])) {
		xenv_error(EX_OSERR, errno, "pipe");
	}
	if (pipe(p[S_ERR])) {
		xenv_error(EX_OSERR, errno, "pipe");
	}

	pid = fork();
	if (pid == -1) {
		xenv_error(EX_OSERR, errno, "fork");
	}

	if (pid == 0) {
		/* Child */
		char *argv[4];
		argv[0] = getenv("SHELL");
		if (!argv[0])
			argv[0] = "/bin/sh";
		argv[1] = "-c";
		argv[2] = xmalloc(len + 1);
		memcpy(argv[2], str, len);
		argv[2][len] = 0;
		argv[3] = NULL;

		close(0);
		if (open("/dev/null", O_RDONLY) == -1) {
			xenv_error(EX_OSERR, errno, "can't open /dev/null");
		}

		if (dup2(p[S_OUT][1], 1) == -1) {
			xenv_error(EX_OSERR, errno, "dup stdout");
		}
		close(p[S_OUT][1]);

		if (dup2(p[S_ERR][1], 2) == -1) {
			xenv_error(EX_OSERR, errno, "dup stderr");
		}
		close(p[S_ERR][1]);

		execvp(argv[0], argv);
		_exit(127);
	} else {
		/* master */
		struct pollfd pfd[2];
		struct timeval end, now;
		int ex;
		int i;
		pid_t pn;
		int sigsent = 0;//FIXME

		close(p[S_OUT][1]);
		close(p[S_ERR][1]);

		gettimeofday(&end, NULL);
		end.tv_sec += command_timeout;

		pfd[S_OUT].fd = p[S_OUT][0];
		pfd[S_OUT].events = POLLIN;
		pfd[S_ERR].fd = p[S_ERR][0];
		pfd[S_ERR].events = POLLIN;

		while (pid != -1 && pfd[S_OUT].fd >= 0 && pfd[S_ERR].fd >= 0) {
			int n, to;
			char buf[BUFSIZ];

			if (pid != -1) {
				pn = waitpid(pid, &ex, WNOHANG);
				if (pn == pid) {
					pid = -1;
				} else if (pn == -1) {
					xenv_error(EX_OSERR, errno, "waitpid");
				}
			}

			if (command_timeout) {
				gettimeofday(&now, NULL);
				if (timeval_cmp(&now, &end) >= 0) {
					error_at_locus(loc,
						       "command timed out");
					kill(pid, SIGKILL);
					sigsent = 1;
					break;
				}
				to = timeval_diff(&end, &now);
			} else
				to = 0;

			n = poll(pfd, 2, to);
			if (n == -1) {
				if (errno == EINTR)
					continue;
				xenv_error(EX_OSERR, errno, "poll");
			}
			for (i = 0; i < S_NUM; i++) {
				if (pfd[i].fd < 0)
					continue;
				else if (pfd[i].revents & POLLIN) {
					n = read(pfd[i].fd, buf, sizeof(buf));
					if (n == -1) {
						xenv_error(EX_OSERR, errno,
							   "read");
					} else if (n == 0) {
						pfd[i].fd = -pfd[i].fd;
					} else if (fn) {
						fn(data, i, buf, n, loc);
					}
				} else if (pfd[i].revents == POLLHUP)
					pfd[i].fd = -pfd[i].fd;
			}
		}

		for (i = 0; i < S_NUM; i++) {
			if (fn)
				fn(data, i, NULL, 0, loc);
			close(p[i][0]);
		}

		if (pid != -1) {
			pn = waitpid(pid, &ex, 0);
			if (pn == -1) {
				xenv_error(EX_OSERR, errno, "waitpid");
			}
		}
		return sigsent ? 0 : ex;
	}
}

struct comlog_status {  /* Command logger status */
	int nl;         /* Newline character was output */
};

/*
 * Log part of command standard output.  Precede each line with the
 * location where command was invoked.
 */
static void
comlog(void *data, int sn, char *str, size_t len, struct locus const *locus)
{
	if (sn == S_ERR) {
		struct comlog_status *st = data;
		while (len) {
			char *p;
			size_t n;

			if (st->nl) {
				locus_print(stderr, locus);
				fprintf(stderr, ": ");
				st->nl = 0;
			}
			if ((p = memchr(str, '\n', len)) != NULL)
				n = p - str + 1;
			else
				n = len;
			fwrite(str, n, 1, stderr);
			len -= n;
			str += n;
		}
	}
}

/*
 * This function is a working horse for ifcom/ifncom.
 * Run command str[0]@len.
 * Return 1 if command exit status is 0.  Otherwise,
 * return 0.
 *
 * Note: If str ends with a '\', assume it is preceded by another backslash
 * and remove it (translate '\\' -> '\').  Scanner rules where this function
 * is invoked ensure that each trailing backslash is escaped.
 */
static int
testcom(char *str, size_t len, struct locus const *loc)
{
	int rc;
	struct comlog_status cls = { 1 };

	if (len > 1 && str[len-1] == '\\')
		len--;
	rc = runcom(str, len, comlog, &cls, loc);
	if (WIFEXITED(rc)) {
		rc = WEXITSTATUS(rc) == 0;
	} else if (WIFSIGNALED(rc)) {
		error_at_locus(loc, "command terminated by signal %d",
			       WTERMSIG(rc));
		exit(EX_OSERR);
	}
	return rc;
}

struct comexp_status { /* Command expansion status. */
	struct comlog_status log; /* Logger status: used for S_ERR. */
	/* Fields below are used for command standard output. */
	size_t nlorig;    /* Original number of newline characters in input. */
	size_t nlout;     /* Actual number of newlines output. */
	size_t nlpending; /* Number of newlines pending for output. */
};

/*
 * Handle standard output and error for a command run by expandcom (see
 * below).
 */
static void
comexp(void *data, int sn, char *str, size_t len,
       struct locus const *locus)
{
	struct comexp_status *st = data;
	if (sn == S_ERR)
		return comlog(data, sn, str, len, locus);
	if (len) {
		size_t n = 0;

		while (len > 0 && str[len-1] == '\n') {
			n++;
			len--;
		}

		if (len > 0) {
			st->nlout += st->nlpending + countnl(str, len);
			while (st->nlpending) {
				fputc('\n', yyout);
				st->nlpending--;
			}
			fwrite(str, len, 1, yyout);
		}
		st->nlpending += n;
	} else if (st->nlorig != st->nlout)
		syncline(1);
}

/*
 * Run the command in STR, redirecting its standard output to yyout.
 * Take care to remove trailing newlines, however.
 * If command_timeout is set, make sure the command is terminated if
 * it runs longer than that.
 */
static void
expandcom(char *str, struct locus const *locus)
{
	int rc;
	struct comexp_status st;
	size_t len = strlen(str);

	st.log.nl = 1;
	st.nlout = 0;
	st.nlorig = countnl(str, len);
	st.nlpending = 0;

	rc = runcom(str, len, comexp, &st, locus);

	if (WIFEXITED(rc) && WEXITSTATUS(rc))
		error_at_locus(locus, "command exited with status %d",
			       WEXITSTATUS(rc));
	else if (WIFSIGNALED(rc)) {
		error_at_locus(locus, "command terminated on signal %d",
			       WTERMSIG(rc));
	}
	beginning_of_line = 0;
}

static void
usage(FILE *fp)
{
	int i;

	fprintf(fp,
		"usage: %s [-hmnrsuvx] [-D NAME[=VALUE]] [-e NAME] [-I DIR] [-o FILE]\n"
		"[-p COMMAND] [-t SECONDS] [-U NAME] [-W [no-]FEATURE]\n",
		progname);
	fprintf(fp, "expands environment variables and shell commands in input files\n");
	fprintf(fp, "\nOPTIONS are:\n\n");
	fprintf(fp, "  -D NAME[=VALUE]  set environment variable NAME\n");
	fprintf(fp, "  -e ENV           define environment meta-variable;\n"
		    "                   variables are to be referenced as $ENV{NAME}\n");
	fprintf(fp, "  -I DIRECTORY     append DIRECTORY to include path\n");
	fprintf(fp, "  -m               pipe output to m4\n");
	fprintf(fp, "  -n               don't produce output, only report errors\n");
	fprintf(fp, "  -o FILE          write output to FILE\n");
	fprintf(fp, "  -p COMMAND       pipe output to COMMAND\n");
	fprintf(fp, "  -r               retain unexpanded constructs in the output\n");
	fprintf(fp, "  -s               generate `#line NUM \"FILE\"' lines\n");
	fprintf(fp, "  -t SECONDS       timeout for command substitution\n");
	fprintf(fp, "  -u               treat unset variables as errors\n");
	fprintf(fp, "  -U NAME          unset environment variable NAME\n");
	fprintf(fp, "  -v               print the version number and exit\n");
	fprintf(fp, "  -W [no-]FEATURE  enable or disable specific xenv feature\n");
	fprintf(fp, "  -x               enable debugging\n");
	fprintf(fp, "  -h or -?         print this help text\n");
	fprintf(fp, "\n");
	fprintf(fp, "FEATUREs are:\n");
	for (i = 0; feature[i].name; i++) {
		fprintf(fp, "  %-16s - %s\n", feature[i].name,
			feature[i].descr);
	}
	fprintf(fp, "\n");

	fprintf(fp, "Report bugs and suggestions to <gray@gnu.org>\n");
#ifdef PACKAGE_URL
	fprintf(fp, "%s home page: <%s>\n", PACKAGE_NAME, PACKAGE_URL);
#endif
}

static int copyright_year = 2025;

static void
version(void)
{
	printf("%s (%s) %s\n", progname, PACKAGE_NAME, PACKAGE_VERSION);
	printf("Copyright (C) 2021-%d Sergey Poznyakoff\n", copyright_year);
	printf("\
License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\nThis is free software: you are free to change and redistribute it.\n\
There is NO WARRANTY, to the extent permitted by law.\n\
");
}

int
main(int argc, char **argv)
{
	int c;
	int dry_run = 0;
	char *pipe_command = NULL;
	char *output_option = NULL;
	int m4_option = 0;

	if ((progname = strrchr(argv[0], '/')) != NULL)
		progname++;
	else
		progname = argv[0];
	yyset_debug(0);
	stash_init();
	while ((c = getopt(argc, argv, "D:e:hI:mno:p:rst:U:uvW:x?")) != EOF) {
		switch (c) {
		case 'D':{
			char *p = strchr(optarg, '=');
			if (p)
				*p++ = 0;
			else
				p = "";
			setenv(optarg, p, 1);
			break;
		}

		case 'I':
			search_path_append(optarg);
			break;

		case 'U':
			unsetenv(optarg);
			break;

		case 'e':
			env_prefix = optarg;
			break;

		case 'h':
			usage(stdout);
			return 0;

		case 'm':
			m4_option = 1;
			break;

		case 'n':
			dry_run = 1;
			break;

		case 'o':
			output_option = optarg;
			break;

		case 'p':
			m4_option = 0;
			pipe_command = optarg;
			break;

		case 'r':
			retain_unexpanded_option = 1;
			break;

		case 's':
			synclines_option = 1;
			break;

		case 't': {
			unsigned long n;
			char *p;

			errno = 0;
			n = strtoul(optarg, &p, 10);
			if (errno || n >= UINT_MAX) {
				xenv_error(EX_USAGE, errno,
					   "invalid timeout value");
			}
			command_timeout = n;
			break;
		}

		case 'u':
			undef_error_option = 1;
			break;

		case 'v':
			version();
			return 0;

		case 'x':
			yyset_debug(1);
			break;

		case 'W':
			if (feature_set(optarg)) {
				xenv_error(EX_USAGE, 0,
					   "invalid feature name: %s", optarg);
			}
			break;

		default:
			if (optopt == '?') {
				usage(stdout);
				return 0;
			}
			usage(stderr);
			return EX_USAGE;
		}
	}

	if (feature_is_set(FEATURE_MINIMAL)) {
		feature_set_minimal();
		if (!env_prefix)
			env_prefix = "ENV";
	} else if (feature_is_set(FEATURE_BOOLEANS) == F_DFL) {
		feature_set("booleans=1/0");
	}

	if (m4_option) {
		if (pipe_command)
			xenv_error(EX_USAGE, 0,
				   "conflicting options: -m and -p");
		if (synclines_option)
			pipe_command = "m4 -s";
		else
			pipe_command = "m4";
	}

	input_files = argv + optind;
	input_index = 0;
	if (input_files[0] == NULL) {
		static char *stdin_file[] = { "-", NULL };
		input_files = stdin_file;
	}

	if (dry_run) {
		if (output_option)
			xenv_error(EX_USAGE, 0,
				   "conflicting options: -n and -o");
		if (pipe_command) {
			int fd = open("/dev/null", O_WRONLY);
			if (fd == -1)
				xenv_error(EX_OSERR, errno,
					   "can't open /dev/null");
			if (fd != 1) {
				if (dup2(fd, 1) == -1)
					xenv_error(EX_OSERR, errno, "dup2");
				close(fd);
			}
		} else
			yyout = sink_open();
	}

	if (pipe_command) {
		if (output_option)
			xenv_error(EX_USAGE, 0,
				   "conflicting options: -p and -o");
		yyout = popen(pipe_command, "w");
		if (!yyout) {
			xenv_error(EX_OSERR, errno, "popen");
		}
	} else if (output_option) {
		if ((yyout = fopen(output_option, "w")) == NULL) {
			int ec = errno;
			xenv_error(0, ec, "can't open output file %s",
				   output_option);
			file_error_exit(ec);
		}
	} else {
		yyout = stdout;
	}

	open_input(input_files[input_index++]);
	yylex();
	if (pipe_command)
		pclose(yyout);
	else
		fclose(yyout);
	if (feature_is_set(FEATURE_PARANOID_MEMFREE)) {
		yylex_destroy();
		xenv_stack_free(&state_stack);
		xenv_stack_free(&expand_inline_stack);
		xenv_stack_free(&cond_stack);
		xenv_stack_free(&paramctx_stack);
		defbool(0, NULL);
		divert_destroy();
		macros_destroy();
		free_filenames();
	}
	return status;
}

#ifndef HAVE_FOPENCOOKIE
FILE *
stash_open(void)
{
	return tmpfile();
}

void
stash_init(void)
{
}

FILE *
sink_open(void)
{
	return fopen("/dev/null", "w+b")
}
#endif
