;;; xenv-mode.el --- major mode for editing xenv preprocessor files

;; Authors: 2025 Sergey Poznyakoff
;; Version: 1.0
;; Keywords: xenv, preprocessor

;; This file is part of xenv
;; Copyright (C) 2021-2025 Sergey Poznyakoff
;;
;; Xenv is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by the
;; Free Software Foundation; either version 3 of the License, or (at your
;; option) any later version.
;;
;; Xenv is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License along
;; with xenv. If not, see <http://www.gnu.org/licenses/>.

;; Installation:
;; You may wish to use precompiled version of the module. To create it
;; run:
;;    emacs -batch -f batch-byte-compile xenv-mode.el
;; Install the file xenv-mode.elc (and, optionally, xenv-mode.el) to
;; any directory in your Emacs load-path.
;;
;; To your .emacs or site-start.el add:
;;  (autoload 'xenv-mode "xenv-mode")

(require 'font-lock)

(defcustom xenv-basic-offset 2
  "*The default indentation increment."
  :type 'integer
  :group 'xenv-indentation)

(defcustom xenv-split-indent t
  "*When indenting directives, place the $$ mark at column 1, inserting
the necessary amount of whitespace between it and the directive."
  :type 'boolean
  :group 'xenv-indentation)

;; This regular expression matches all xenv keywords.
(defconst xenv-keywords
  (regexp-opt '("defmacro" "divert" "dnl" "dropdivert" "else" "end"
		"endif" "error" "eval" "evaldivert" "exit" "expmacro"
		"ifcom" "ifdef" "iffalse" "ifncom" "ifndef" "ifnset"
		"ifset" "iftrue" "ignore" "include" "loop" "range"
		"require" "set" "sinclude" "source" "undivert" "unset"
		"verbatim" "warning")))

;; Regular expression that matches xenv "if*" directives.
;; In other words, directives, that end with "$$endif".
(defconst xenv-if-keywords
  (regexp-opt '("ifcom" "ifdef" "iffalse" "ifncom" "ifndef" "ifnset"
		"ifset" "iftrue")))

;; Regular expression that matches xenv keywords that are followed by
;; a variable name.
(defconst xenv-param-keywords
  (regexp-opt '("ifdef" "iffalse" "ifndef"
		"ifnset" "ifset" "iftrue" "loop" "range" "set" "unset")))

;; Regexp matching xenv directives that end with a "$$end" and allow
;; for nested directives.
(defconst xenv-begin-keywords
  (regexp-opt '("defmacro" "eval" "loop" "range")))

(defconst xenv-open-keywords
  (regexp-opt '("defmacro" "eval" "ifcom" "ifdef" "iffalse" "ifncom"
		"ifndef" "ifnset" "ifset" "iftrue" "loop" "range")))


;; Regexp matching xenv directives that end with a "$$end" and don't
;; allow for nested directives.
(defconst xenv-simple-begin-keywords
  (regexp-opt '("ignore" "verbatim")))

(defconst xenv-directive-prefix-re "^[ \t]*\\$\\$[ \t]*")
(defconst xenv-variable-re "[[:alpha:]_][[:alnum:]_]*")

(defconst xenv-syntax-propertize-function
  (syntax-propertize-rules
   ("^[ \t]*\\(\\$\\$\\)[ \t]*dnl\\>" (1 "<"))
   ("^[ \t]*\\(\\$\\$\\)[ \t]*ignore\\>" (1 "< b"))
   ("^[ \t]*\\(\\$\\$\\)[ \t]*end[ \t]*\\(\n\\)" (2 "> b"))
   ("\\${\\*" (0 "< b"))
   ("\\*\\(}\\)" (1 "> b")))
  "Syntactic keywords to catch comment delimiters in `xenv-mode'.")

(defvar xenv-mode-syntax-table
  (let ((st (make-syntax-table)))
    (modify-syntax-entry ?\' "\"" st)
    (modify-syntax-entry ?\" "\"" st)
    (modify-syntax-entry ?\\ "\\" st)
    st))

;; Font-lock stuff
(defconst xenv-font-lock-keywords
  (list
   ;; Fontify keywords
   (list
    (concat "^[ \t]*\\$\\$[ \t]*\\(" xenv-keywords "\\)\\>")
    '(1 font-lock-builtin-face))

   ;; Fontify parameter names
   (list
    (concat xenv-directive-prefix-re xenv-param-keywords
	    "[ \t]+\\(" xenv-variable-re "\\)")
    '(1 font-lock-variable-name-face))
   (list
    (concat xenv-directive-prefix-re "defmacro[ \t]+")
    xenv-variable-re
    nil nil
    '(0 font-lock-variable-name-face))

   ;; Variable names
   '("\\${?\\([[:alpha:]_][[:alnum:]_]*\\)" 1 font-lock-variable-name-face) ))

(defun xenv-find-indent-start ()
  "Move point to the start location for computing the indentation level
of the current line.  Return its position in buffer."
  (let ((pos (point))
	(bod (save-excursion (xenv-set-beginning-of-defun))))
    ;; Check if we're within defmacro.  If so, return the point of
    ;; its beginning
    (if bod
	(let ((eod (save-excursion
		     (forward-line 1)
		     (xenv-find-matching-end))))
	  (if (and eod (> eod pos))
	      bod
	    0))
      0)))

(defun xenv-indent-level ()
  "Return a cons (L . C), where L is the nesting (indentation) level, and
C indicates context for the current line: nil, for top-level, 0 for block
statement (defmacro, loop, etc), and 1 for conditional statement."
  (save-excursion
    (let ((ctx nil)
	  (stop (progn
		  (beginning-of-line)
		  (point))))
      (goto-char (xenv-find-indent-start))
      (catch 'ret
	(while (< (point) stop)
	  (cond
	   ((looking-at
	     (concat xenv-directive-prefix-re xenv-if-keywords "\\>"))
	    (setq ctx (cons 1 ctx)))
	   ((looking-at
	     (concat xenv-directive-prefix-re xenv-begin-keywords "\\>"))
	    (setq ctx (cons 0 ctx)))
	   ((looking-at
	     (concat xenv-directive-prefix-re xenv-simple-begin-keywords "\\>"))
	    (if (not (re-search-forward
		      (concat xenv-directive-prefix-re "end\\>") nil t))
		(setq ctx nil)
		(throw 'ret t)))
	   ((looking-at (concat xenv-directive-prefix-re "end\\>"))
	    (cond ((and ctx (= 0 (car ctx)))
		   (setq ctx (cdr ctx)))))
	   ((looking-at (concat xenv-directive-prefix-re "endif\\>"))
	    (cond ((and ctx (= 1 (car ctx)))
		   (setq ctx (cdr ctx))))))
	  (forward-line 1)))
      (let ((level (length ctx)))
	(if (> level 0)
	    (cons level (car ctx))
	  (cons level nil))))))

(defun xenv-get-line-indent (kw)
  "Return indent for the current line.  KW is the first xenv keyword
on the line."
  (let ((level (xenv-indent-level)))
    (* xenv-basic-offset
       (cond
	((not (cdr level))
	 (car level))
	((and (= 1 (cdr level)) (or (string= kw "endif") (string= kw "else")))
	 (1- (car level)))
	((and (= 0 (cdr level)) (string= kw "end"))
	 (1- (car level)))
	(t
	 (car level))))))

(defun xenv-current-indentation ()
  "Return cons: the indentation of the current line and boolean indicating
if it doesn't match the xenv-split-indent.  The car differs from what
(current-indentation) returns in that it correctly handles lines starting
with $$."
  (let ((xstmt (save-excursion
		 (beginning-of-line)
		 (looking-at "^\\([ \t]*\\)\\$\\$\\([ \t]*\\)"))))
    (if xstmt
	(let ((p (not (string= (match-string 1) "")))
	      (s (not (string= (match-string 2) ""))))
	  (cons
	   (if p
	       (current-indentation)
	       (string-width (match-string 2)))
	   (or
	    (and p s)
	    (if xenv-split-indent p s))))
      (cons
       (current-indentation)
       nil))))

(defun xenv-indent-line ()
  "Indent the current line."
  (interactive "*")
  (if (save-excursion
	(beginning-of-line)
	(looking-at (concat xenv-directive-prefix-re "\\("
			    xenv-keywords "\\)")))
      (let* ((line-start (match-beginning 1))
	     (indent (xenv-get-line-indent (match-string 1)))
	     (cur (xenv-current-indentation))
	     (delta (- indent (car cur))))
	(if (or (not (= delta 0)) (cdr cur))
	    (let ((off (- (point) line-start)))
	      (beginning-of-line)
	      (delete-region (point) line-start)
	      (indent-to indent)
	      (if xenv-split-indent
		  (beginning-of-line))
	      (insert "$$")
	      (if (>= off 0	)
		  (goto-char (+ line-start delta off))
		(beginning-of-line)))))))

(defun xenv-newline-and-indent ()
  "Indent the current line, if necessary, insert a newline, and then
indent again.
"
  (interactive "*")
  (if (save-excursion
	(beginning-of-line)
	(looking-at (concat xenv-directive-prefix-re "\\("
			    xenv-keywords "\\)")))
      (xenv-indent-line))
  (newline-and-indent))

(defun xenv-set-beginning-of-defun ()
  "Sets point to the start of the nearest previous defmacro statement,
or to the beginning of buffer, if no such statement was found."
  (catch 'loop
    (while (not (bobp))
      (forward-line -1)
      (if (looking-at (concat xenv-directive-prefix-re "defmacro[ \t]+"))
	  (throw 'loop (point))))
    nil))

(defun xenv-beginning-of-defun ()
  "Interface to `beginning-of-defun'"
  (let ((pos (save-excursion (xenv-set-beginning-of-defun))))
    (cond
     (pos
      (push-mark)
      (goto-char pos)))))

(defun xenv-find-matching-end ()
  "Move point to the line containig the closing $$end for the current
block construct."
  (catch 'loop
    (let ((level 0))
      (while (not (eobp))
	(cond
	 ((looking-at
	   (concat xenv-directive-prefix-re xenv-simple-begin-keywords "\\>"))
	  (re-search-forward (concat xenv-directive-prefix-re "end\\>")))
	 ((looking-at
	   (concat xenv-directive-prefix-re xenv-begin-keywords "\\>"))
	  (setq level (1+ level)))
	 ((looking-at
	   (concat xenv-directive-prefix-re "end\\>"))
	  (if (= level 0)
	      (throw 'loop (point))
	    (setq level (1- level)))))
	(forward-line 1)))
    nil))

(defun xenv-find-matching-endif ()
  (catch 'loop
    (let ((level 0))
      (while (not (eobp))
	(cond
	 ((looking-at
	   (concat xenv-directive-prefix-re xenv-if-keywords "\\>"))
	  (setq level (1+ level)))
	 ((looking-at
	   (concat xenv-directive-prefix-re "endif\\>"))
	  (if (= level 0)
	      (throw 'loop (point))
	    (setq level (1- level)))))
	(forward-line 1)))
    nil))

(defun xenv-find-if ()
  (catch 'loop
    (forward-line -1)
    (let ((level 0))
      (while (not (bobp))
	(cond
	 ((looking-at
	   (concat xenv-directive-prefix-re "endif\\>"))
	  (setq level (1+ level)))
	 ((looking-at
	   (concat xenv-directive-prefix-re xenv-if-keywords "\\>"))
	  (if (= level 0)
	      (throw 'loop (point))
	    (setq level (1- level)))))
	(forward-line -1))
      nil)))

(defun xenv-find-open ()
  (catch 'loop
    (forward-line -1)
    (let ((level 0))
      (while (not (bobp))
	(cond
	 ((looking-at
	   (concat xenv-directive-prefix-re "end\\>"))
	  (setq level (1+ level)))
	 ((or
	   (looking-at
	    (concat xenv-directive-prefix-re xenv-begin-keywords "\\>"))
	   (looking-at
	    (concat xenv-directive-prefix-re xenv-simple-begin-keywords "\\>")))
	  (if (= level 0)
	      (throw 'loop (point))
	    (setq level (1- level)))))
	(forward-line -1))
      nil)))

(defun xenv-end-of-defun ()
  "Interface to `end-of-defun'"
  (let ((pos (save-excursion
	       (and (xenv-set-beginning-of-defun)
		    (forward-line 1)
		    (xenv-find-matching-end)))))
    (cond
     (pos
      (push-mark)
      (goto-char pos)))))

(defun xenv-beginning-of-construct ()
  (interactive "*")
  (let ((pos
	 (save-excursion
	   (beginning-of-line)
	   (cond
	    ((looking-at (concat xenv-directive-prefix-re "endif\\>"))
	     (xenv-find-if))
	    ((looking-at (concat xenv-directive-prefix-re "end\\>"))
	     (xenv-find-open))
	    (t
	     (re-search-backward
	      (concat xenv-directive-prefix-re xenv-open-keywords "\\>")
	      nil t))))))
    (cond
     (pos
      (push-mark)
      (goto-char pos)))))

(defun xenv-end-of-construct ()
  (interactive "*")
  (let ((re (concat xenv-directive-prefix-re "\\("
		    xenv-open-keywords "\\)\\>")))
    (let ((pos
	   (save-excursion
	     (beginning-of-line)
	     (cond
	      ((or (looking-at re)
		   (re-search-backward re nil t))
	       (forward-line 1)
	       (let ((kw (match-string 1)))
		 (if (string-match xenv-if-keywords kw)
		     (xenv-find-matching-endif)
		   (xenv-find-matching-end))))))))
      (cond
       (pos
	(push-mark)
	(goto-char pos))))))

(defvar xenv-mode-map
  (let ((map (make-sparse-keymap)))
    (define-key map "\C-\M-a" 'beginning-of-defun)
    (define-key map "\C-\M-e" 'end-of-defun)
    (define-key map "\t" 'xenv-indent-line)
    (define-key map "\r" 'xenv-newline-and-indent)
    (define-key map "\C-\M-f" 'xenv-end-of-construct)
    (define-key map "\C-\M-b" 'xenv-beginning-of-construct)
    map))

;;;###autoload
(define-derived-mode xenv-mode text-mode "Xenv"
  "Major mode for editing xenv preprocessor files.

Key bindings are:
\\{xenv-mode-map}
"

  (set-syntax-table xenv-mode-syntax-table)
  (use-local-map xenv-mode-map)

  (setq-local beginning-of-defun-function 'xenv-beginning-of-defun)
  (setq-local end-of-defun-function 'xenv-end-of-defun)
  (setq-local syntax-propertize-function xenv-syntax-propertize-function)
  (setq-local indent-tabs-mode nil)
  (setq-local indent-line-function 'xenv-indent-line)
  (setq-local font-lock-defaults
	'((xenv-font-lock-keywords) nil nil
	  nil
	  nil)))

(provide 'xenv-mode)
;;; xenv-mode ends
