/* This file is part of mbar.
   Copyright (C) 2016, 2019, 2025 Sergey Poznyakoff

   Mbar is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   Mbar is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mbar.  If not, see <http://www.gnu.org/licenses/>. */

#include "config.h"
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <mailutils/mailutils.h>
#include <mailutils/nls.h>
#include <sysexits.h>
#include <sys/stat.h>
#include <stdarg.h>
#include <regex.h>
#include <stdint.h>

#define PACKAGE_URL  "https://puszcza.gnu.org.ua/software/mbar"
#define PACKAGE_BUGREPORT "gray@gnu.org.ua"

/* FIXME */
#define _(string) string
#define N_(string) string
#ifndef MU_APP_INIT_NLS
# define MU_APP_INIT_NLS()
#endif

int verbose;
int action_trace;
int max_depth = 0;
int dry_run;
int compile_only;
int locus_option;
mu_list_t pattern, exclude;
enum
  {
    PAT_GLOB,
    PAT_IMAP,
    PAT_REGEX
  };
int pattern_type = PAT_GLOB;
char *sourcedir;
char *destdir;
mu_list_t require_list;
char *require_string;
char *template_file;
char *mailbox_type;
int keep_option;
mu_list_t env_list;
mu_list_t var_list;
mu_list_t cond_list;

#define EX_OK 0

uintmax_t dir_count;
uintmax_t mbx_count;

int stat_option;
uintmax_t msg_total;
uintmax_t size_total;
uintmax_t msg_archived;
uintmax_t size_archived;


void
abfun (int code, char const *func, char const *arg, int rc)
{
  mu_diag_funcall (MU_DIAG_ERROR, func, arg, rc);
  exit (code);
}

MU_PRINTFLIKE(2,3) void
abend (int code, char const *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  mu_diag_vprintf (MU_DIAG_ERROR, fmt, ap);
  va_end (ap);
  mu_diag_printf (MU_DIAG_ERROR, "\n");
  exit (code);
}

struct segment
{
  char *beg;
  char *end;
  char *cur;
};

static char *
list_to_string (mu_list_t list, mu_list_action_t cb_size,
		mu_list_action_t cb_cat, char *pfx, char *sfx)
{
  size_t size = 0, total;
  char *retval;
  struct segment segm;

  if (mu_list_is_empty (list))
    return NULL;

  mu_list_foreach (list, cb_size, &size);
  total = size;
  if (pfx)
    total += strlen (pfx);
  if (sfx)
    total += strlen (sfx);
  retval = mu_alloc (total + 1);
  segm.beg = retval;
  if (pfx)
    segm.beg = mu_stpcpy (segm.beg, pfx);
  segm.cur = segm.beg;
  segm.end = segm.cur + size;
  mu_list_foreach (list, cb_cat, &segm);
  if (sfx)
    segm.cur = mu_stpcpy (segm.cur, sfx);
  *segm.cur = 0;
  mu_list_destroy (&list);
  return retval;
}

static int
req_size (void *item, void *data)
{
  char *str = item;
  size_t *sz = data;
  *sz += strlen (str) + 3;
  return 0;
}

static int
req_concat (void *item, void *data)
{
  char *str = item;
  struct segment *segm = data;

  *segm->cur++ = ',';
  *segm->cur++ = '"';
  segm->cur = mu_stpcpy (segm->cur, str);
  *segm->cur++ = '"';

  return 0;
}

static void
build_requires (void)
{
  if (require_list)
    require_string = list_to_string (require_list, req_size, req_concat,
				     NULL, NULL);
}

static int
cond_size (void *item, void *data)
{
  char *str = item;
  size_t *sz = data;
  *sz += strlen (str) + 1;
  return 0;
}

static int
cond_concat (void *item, void *data)
{
  char *str = item;
  struct segment *segm = data;

  if (segm->cur > segm->beg)
    *segm->cur++ = ',';
  segm->cur = mu_stpcpy (segm->cur, str);
  return 0;
}

static char *
build_cond (void)
{
  if (!cond_list)
    return mu_strdup ("true");
  return list_to_string (cond_list, cond_size, cond_concat,
			 "allof(", ")");
}

char *prog_template = "require [\"fileinto\"${requires}];\n\
if ${cond} {\n\
    fileinto :interdir \"\\${dest}\";${keep:+keep;}\n\
}\n";

void
read_template_file (void)
{
  if (template_file)
    {
      mu_stream_t ins;
      mu_off_t size;
      int rc;

      rc = mu_file_stream_create (&ins, template_file, MU_STREAM_READ);
      if (rc)
	abfun (EX_OSERR, "mu_file_stream_create", template_file, rc);

      mu_stream_size (ins, &size);

      prog_template = mu_alloc (size + 1);

      rc = mu_stream_read (ins, prog_template, size, NULL);
      if (rc)
	abfun (EX_OSERR, "mu_stream_read", template_file, rc);
      prog_template[size] = 0;

      mu_stream_destroy (&ins);
    }
}

static void
_sieve_action_log (mu_sieve_machine_t mach,
		   const char *action, const char *fmt, va_list ap)
{
  size_t uid = 0;
  mu_stream_t stream;
  mu_message_t msg = mu_sieve_get_message (mach);
  char *folder = mu_sieve_get_data (mach);

  mu_sieve_get_diag_stream (mach, &stream);

  mu_message_get_uid (msg, &uid);
  mu_stream_printf (stream, "\033s<%d>\033%c<%d>", MU_LOG_NOTICE,
		    locus_option ? 'O' : 'X', MU_LOGMODE_LOCUS);
  mu_stream_printf (stream, _("%s on %s:%zu"), action, folder, uid);

  if (fmt && strlen (fmt))
    {
      mu_stream_printf (stream, ": ");
      mu_stream_vprintf (stream, fmt, ap);
    }
  mu_stream_printf (stream, "\n");
}

static void
_stat_action_log (mu_sieve_machine_t mach,
		  const char *action, const char *fmt, va_list ap)
{
  if (action_trace || verbose > 3)
    _sieve_action_log (mach, action, fmt, ap);
  if (strcmp (action, "FILEINTO") == 0)
    {
      size_t size;
      msg_archived++;
      if (mu_message_size (mu_sieve_get_message (mach), &size) == 0)
	size_archived += size;
    }
}

static void
nl (mu_stream_t stream, char const *script)
{
  unsigned line = 0;
  while (*script)
    {
      size_t len = strcspn (script, "\n");
      ++line;
      mu_stream_printf (stream, "%4u: ", line);
      mu_stream_write (stream, script, len, NULL);
      mu_stream_write (stream, "\n", 1, NULL);
      script += len;
      if (*script)
	++script;
    }
}

static int
sieve_setenv (void *item, void *data)
{
  char *str = item;
  mu_sieve_machine_t mach = data;
  int rc = mu_sieve_set_environ (mach, str, str + strlen (str) + 1);
  if (rc)
    mu_error (_("can't set environment item %s: %s"),
	      str, mu_strerror (rc));
  return 0;
}

static int
sieve_setvar (void *item, void *data)
{
  char *str = item;
  mu_sieve_machine_t mach = data;
  mu_sieve_variable_initialize (mach, str, str + strlen (str) + 1);
  return 0;
}

static mu_sieve_machine_t
build_sieve (char *cond, int show_script)
{
  struct mu_wordsplit ws;
  char const *env[7];
  int rc;
  mu_sieve_machine_t mach;
  struct mu_locus_point loc = MU_LOCUS_POINT_INITIALIZER;

  env[0] = "cond";
  env[1] = cond;

  env[2] = "requires";
  env[3] = require_string ? require_string : "";

  env[4] = "keep";
  env[5] = keep_option ? "1" : "";

  env[6] = NULL;

  ws.ws_env = env;
  if (mu_wordsplit (prog_template, &ws,
		    MU_WRDSF_NOCMD
		    | MU_WRDSF_ENV
		    | MU_WRDSF_ENV_KV
		    | MU_WRDSF_NOSPLIT
		    | MU_WRDSF_SHOWERR
		    | MU_WRDSF_WARNUNDEF))
    exit (EX_CONFIG);

  if (show_script)
    {
      mu_printf ("Sieve script:\n");
      nl (mu_strout, ws.ws_wordv[0]);
    }

  /* Sieve interpreter setup. */
  rc = mu_sieve_machine_create (&mach);
  if (rc)
    abend (EX_SOFTWARE, _("cannot initialize sieve machine: %s"),
	   mu_strerror (rc));

  if (stat_option)
    mu_sieve_set_logger (mach, _stat_action_log);
  else if (action_trace || verbose > 3)
    mu_sieve_set_logger (mach, _sieve_action_log);

  mu_sieve_set_environ (mach, "location", "MS");
  mu_sieve_set_environ (mach, "phase", "post");

  mu_list_foreach (env_list, sieve_setenv, mach);
  mu_list_destroy (&env_list);

  mu_sieve_require_variables (mach);
  mu_list_foreach (var_list, sieve_setvar, mach);
  mu_list_destroy (&var_list);
  mu_sieve_variable_initialize (mach, "keep", keep_option ? "1" : "");

  mu_sieve_set_dry_run (mach, dry_run);

  mu_locus_point_set_file (&loc, "stdin");
  loc.mu_line = 1;

  rc = mu_sieve_compile_text (mach, ws.ws_wordv[0], strlen (ws.ws_wordv[0]),
			      &loc);
  mu_locus_point_deinit (&loc);

  if (rc)
    {
      if (!show_script)
	{
	  mu_stream_printf (mu_strerr, "%s", _("Failed program was:\n"));
	  nl (mu_strerr, ws.ws_wordv[0]);
	}
      exit (EX_CONFIG);
    }
  mu_wordsplit_free (&ws);

  return mach;
}

static mu_sieve_machine_t
mach_setup (mu_sieve_machine_t mach, char const *folder)
{
  mu_sieve_machine_t clone;
  int rc;
  char *dest;
  char const *p = folder + strlen (sourcedir);
  while (p[0] == '/')
    p++;
  dest = mu_make_file_name (destdir, p);
  rc = mu_sieve_machine_clone (mach, &clone);
  if (rc)
    abfun (EX_OSERR, "mu_sieve_machine_clone", NULL, rc);
  mu_sieve_set_data (clone, (void*) folder);
  mu_sieve_variable_initialize (clone, "source", folder);
  mu_sieve_variable_initialize (clone, "dest", dest);
  free (dest);
  return clone;
}

static int
real_switch_to_user (char const *path)
{
  struct stat st;

  if (stat (path, &st))
    {
      mu_diag_funcall (MU_DIAG_ERROR, "stat", path, errno);
      return -1;
    }
  if (setregid (0, st.st_gid))
    {
      mu_error (_("%s: failed to switch to gid %lu: %s"),
		path, (unsigned long)st.st_gid, mu_strerror (errno));
      return -1;
    }

  if (setreuid (0, st.st_uid))
    {
      mu_error (_("%s: failed to switch to gid %lu: %s"),
		path, (unsigned long)st.st_gid, mu_strerror (errno));
      if (setregid (0, 0))
	{
	  mu_error (_("additionally, failed to restore root gid: %s"),
		    mu_strerror (errno));
	  mu_error ("that's fatal");
	  exit (EX_OSERR);
	}
    }
  return 0;
}

static void
real_switch_to_root (void)
{
  if (setregid (0, 0))
    abend (EX_OSERR, _("failed to restore root gid: %s"), mu_strerror (errno));
  if (setreuid (0, 0))
    abend (EX_OSERR, _("failed to restore root uid: %s"), mu_strerror (errno));
}

static int
dummy_switch_to_user (char const *path)
{
  return 0;
}

static void
dummy_switch_to_root (void)
{
}

static int (*switch_to_user) (char const *) = dummy_switch_to_user;
static void (*switch_to_root) (void) = dummy_switch_to_root;

static int
glob_patmat (void *pattern, char const *name)
{
  return mu_folder_glob_match (name, pattern, '/');
}

static int
imap_patmat (void *pattern, char const *name)
{
  return mu_folder_glob_match (name, pattern, '/');
}

static int
regex_patmat (void *pattern, char const *name)
{
  return regexec (pattern, name, 0, NULL, 0);
}

static int (*patmat) (void *, char const *) = glob_patmat;

static void
list_regcomp (mu_list_t list)
{
  mu_iterator_t itr;

  if (mu_list_is_empty (list))
    return;
  mu_list_get_iterator (list, &itr);
  for (mu_iterator_first (itr); !mu_iterator_is_done (itr);
       mu_iterator_next (itr))
    {
      char *pattern;
      int rc;
      char errbuf[BUFSIZ];
      regex_t *rx = mu_alloc (sizeof (*rx));

      mu_iterator_current (itr, (void**)&pattern);

      rc = regcomp (rx, pattern, REG_EXTENDED);
      if (rc)
	{
	  regerror (rc, rx, errbuf, sizeof (errbuf));
	  abend (EX_USAGE, "%s", errbuf);
	}
      mu_iterator_ctl (itr, mu_itrctl_replace, rx);
    }
  mu_iterator_destroy (&itr);
}

int
list_patmat (mu_list_t list, char const *name)
{
  int res = 1;
  if (!mu_list_is_empty (list))
    {
      mu_iterator_t itr;

      mu_list_get_iterator (list, &itr);
      for (mu_iterator_first (itr); !mu_iterator_is_done (itr);
	   mu_iterator_next (itr))
	{
	  void *pattern;
	  mu_iterator_current (itr, &pattern);
	  if (patmat (pattern, name) == 0)
	    {
	      res = 0;
	      break;
	    }
	}
      mu_iterator_destroy (&itr);
    }
  return res;
}

static int
enumfun (mu_folder_t folder, struct mu_list_response *resp, void *data)
{
  mu_sieve_machine_t mach = data;
  mu_mailbox_t mbox;
  int rc;
  mu_url_t url;
  char const *path;
  int matched = 0;

  if (resp->type & MU_FOLDER_ATTRIBUTE_DIRECTORY)
    dir_count++;

  matched = (resp->type & MU_FOLDER_ATTRIBUTE_FILE) &&
    (!pattern || list_patmat (pattern, resp->name) == 0) &&
    (!exclude || list_patmat (exclude, resp->name) != 0);

  if (verbose)
    {
      if (verbose > 1 || matched)
	mu_printf ("%c%c%c %4d %s\n",
		   (resp->type & MU_FOLDER_ATTRIBUTE_DIRECTORY) ? 'd' : '-',
		   (resp->type & MU_FOLDER_ATTRIBUTE_FILE) ? 'm' : '-',
		   matched ? '+' : '-',
		   resp->depth,
		   resp->name);
    }

  if (!matched)
    return 0;

  mbx_count++;

  rc = mu_mailbox_create_at (&mbox, folder, resp->name);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_create", resp->name, rc);
      return 0;
    }

  rc = mu_mailbox_open (mbox, dry_run ? MU_STREAM_READ : MU_STREAM_RDWR);
  if (rc)
    abfun (EX_OSERR, "mu_mailbox_open", resp->name, rc);

  mu_mailbox_get_url (mbox, &url);
  mu_url_sget_path (url, &path);

  switch_to_user (path);

  if (!mailbox_type)
    {
      char const *scheme;
      rc = mu_url_sget_scheme (url, &scheme);
      if (rc)
	{
	  mu_diag_funcall (MU_DIAG_ERROR, "mu_url_sget_scheme", NULL, rc);
	  mu_mailbox_destroy (&mbox);
	  goto end;
	}
      rc = mu_registrar_set_default_scheme (scheme);
      if (rc)
	{
	  mu_diag_funcall (MU_DIAG_ERROR, "mu_registrar_set_default_scheme",
			   scheme, rc);
	  mu_mailbox_destroy (&mbox);
	  goto end;
	}
    }

  mach = mach_setup (mach, path);

  rc = mu_sieve_mailbox (mach, mbox);

  if (stat_option)
    {
      mu_off_t size;
      size_t n;
      if (mu_mailbox_get_size (mbox, &size) == 0)
	size_total += size;
      if (mu_mailbox_messages_count (mbox, &n) == 0)
	msg_total += n;
    }

  if (!dry_run)
    {
      rc = mu_mailbox_expunge (mbox);
      if (rc)
	mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_expunge",
			 resp->name, rc);
    }

  mu_mailbox_close (mbox);
  mu_mailbox_destroy (&mbox);
  mu_sieve_machine_destroy (&mach);
 end:
  switch_to_root ();

  return 0;
}

int
scan_folders (mu_sieve_machine_t mach)
{
  int status;
  mu_folder_t folder;

  status = mu_folder_create (&folder, sourcedir);
  if (status)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_folder_create", sourcedir, status);
      return 1;
    }

  status = mu_folder_open (folder, MU_STREAM_READ);
  if (status)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_folder_open", sourcedir, status);
      return 1;
    }

  status = mu_folder_enumerate (folder, NULL, NULL, 0, max_depth,
				NULL, enumfun, mach);

  mu_folder_destroy (&folder);

  switch (status)
    {
    case 0:
      if (verbose || stat_option)
	mu_printf (_("scanned %ju mailboxes in %ju folders\n"),
		   mbx_count, dir_count + 1);
      if (stat_option)
	{
	  mu_printf (_("messages: %ju/%ju\n"), msg_archived, msg_total);
	  mu_printf (_("size: %ju/%ju\n"), size_archived, size_total);
	}
      break;

    case MU_ERR_NOENT:
      mu_printf (_("No folders matching pattern in %s\n"), sourcedir);
      return EX_OK;

    default:
      mu_error (_("mu_folder_list failed: %s"), mu_strerror (status));
      return EX_UNAVAILABLE;
    }
  return EX_OK;
}

static void
memlist_append (mu_list_t *plist, char *str)
{
  if (!*plist)
    {
      mu_list_create (plist);
      mu_list_set_destroy_item (*plist, mu_list_free_item);
    }
  mu_list_append (*plist, str);
}

static int
list_streq (const void *a, const void *b)
{
  return strcmp (a, b);
}

static void
add_require (const char *arg)
{
  if (!require_list)
    {
      mu_list_create (&require_list);
      mu_list_set_destroy_item (require_list, mu_list_free_item);
      mu_list_set_comparator (require_list, list_streq);
    }
  else if (mu_list_locate (require_list, (void*) arg, NULL) == 0)
    return;
  mu_list_append (require_list, mu_strdup (arg));
}

static void
cond_add_timestamp (int before, char const *date)
{
  char *timecond;
  time_t tlimit, now = time (NULL);

  if (mu_parse_date (date, &tlimit, &now))
    abend (EX_USAGE, _("bad date specification"));

  add_require ("test-timestamp");
  mu_asprintf (&timecond, "timestamp :%s \"Date\" \"%s\"",
	       before ? "before" : "after", date);
  memlist_append (&cond_list, timecond);
}


static void
modify_debug_flags (mu_debug_level_t set, mu_debug_level_t clr)
{
  mu_debug_level_t lev;

  mu_debug_get_category_level (mu_sieve_debug_handle, &lev);
  mu_debug_set_category_level (mu_sieve_debug_handle, (lev & ~clr) | set);
}

static void
cli_instr (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  modify_debug_flags (MU_DEBUG_LEVEL_MASK(MU_DEBUG_TRACE9), 0);
}

static void
cli_require (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  add_require (arg);
}

static void
assign (struct mu_parseopt *po, struct mu_option *opt, char const *arg,
	mu_list_t *plist, char const *what)
{
  char *p = strchr (arg, '=');
  if (p == NULL)
    mu_parseopt_error (po, _("malformed %s: %s"), what, arg);
  else
    {
      char *str;

      str = mu_strdup (arg);
      str[p - arg] = 0;
      memlist_append (plist, str);
    }
}

static void
cli_env (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  assign (po, opt, arg, &env_list, _("environment setting"));
}

static void
cli_var (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  assign (po, opt, arg, &var_list, _("variable assignment"));
}

static void
cli_before (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  cond_add_timestamp (1, arg);
}

static void
cli_after (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  cond_add_timestamp (0, arg);
}

static void
cli_pattern_type (struct mu_parseopt *po, struct mu_option *opt,
		  char const *arg)
{
  if (strcmp (arg, "glob") == 0)
    pattern_type = PAT_GLOB;
  else if (strcmp (arg, "imap") == 0)
    pattern_type = PAT_IMAP;
  else if (strcmp (arg, "regex") == 0)
    pattern_type = PAT_REGEX;
  else
    mu_parseopt_error (po, _("unsupported pattern type"));
}

static void
pattern_from_file (mu_list_t *plist, char const *filename)
{
  mu_stream_t in;
  int rc;
  char *buf = NULL, *p;
  size_t size = 0, n;

  rc = mu_file_stream_create (&in, filename, MU_STREAM_READ);
  if (rc)
    abend (rc == ENOENT ? EX_USAGE : EX_OSERR, "%s: %s", filename,
	   mu_strerror (rc));
  while ((rc = mu_stream_getline (in, &buf, &size, &n)) == 0 && n > 0)
    {
      p = mu_str_stripws (buf);
      if (p[0] && p[0] != '#')
	{
	  if (*p == '\\' && p[1] == '#')
	    p++;
	  memlist_append (plist, mu_strdup (p));
	}
    }
  mu_stream_destroy (&in);
  if (rc)
    abend (EX_UNAVAILABLE, "%s: %s", filename, mu_strerror (rc));
}

static void
cli_pattern (struct mu_parseopt *po, struct mu_option *opt, char const *arg)
{
  if (*arg == '@')
    pattern_from_file (opt->opt_ptr, arg + 1);
  else
    memlist_append (opt->opt_ptr, mu_strdup (arg));
}

static int copyright_year = 2025;
static char gplv3[] = "\
License GPLv3+: GNU GPL version 3 or later <http://gnu.org/licenses/gpl.html>\nThis is free software: you are free to change and redistribute it.\n\
There is NO WARRANTY, to the extent permitted by law.\n\
";

static void
version_hook (struct mu_parseopt *po, mu_stream_t str)
{
  mu_stream_printf (str, "%s (%s) %s\n",
		    po->po_prog_name, PACKAGE_NAME, PACKAGE_VERSION);
  mu_stream_printf (str, "Copyright (C) 2016-%d Sergey Poznyakoff\n",
		    copyright_year);
  mu_stream_printf (str, "%s", gplv3);
#if MAILUTILS_VERSION_PATCH > 0
  mu_stream_printf (str, "\nLinked with GNU Mailutils %d.%d.%d.\n",
		    MAILUTILS_VERSION_MAJOR, MAILUTILS_VERSION_MINOR,
		    MAILUTILS_VERSION_PATCH);
#else
  mu_stream_printf (str, "\nLinked with GNU Mailutils %d.%d.\n",
		    MAILUTILS_VERSION_MAJOR, MAILUTILS_VERSION_MINOR);
#endif
}

/* mbar [options] DIR DEST [EXPR] */
struct mu_option mbar_options[] = {
  MU_OPTION_GROUP("Operation mode selection"),
  { "compile-only", 'c', NULL, MU_OPTION_DEFAULT,
    N_("compile script and exit"),
    mu_c_bool, &compile_only },
  { "dump", 'D', NULL, MU_OPTION_DEFAULT,
    N_("compile script, dump disassembled sieve code to terminal and exit"),
    mu_c_int, &compile_only, NULL, "2" },

  MU_OPTION_GROUP("Basic conditions"),
  { "before-date", 'b', N_("DATE"), MU_OPTION_DEFAULT,
    N_("archive messages received before DATE"),
    mu_c_string, NULL, cli_before },
  { "after-date", 'a', N_("DATE"), MU_OPTION_DEFAULT,
    N_("archive messages received after DATE"),
    mu_c_string, NULL, cli_after },

  MU_OPTION_GROUP("Modifiers"),
  { "mailbox-type", 0, N_("TYPE"), MU_OPTION_DEFAULT,
    N_("set type of destination mailboxes"),
    mu_c_string, &mailbox_type },
  { "maxdepth", 'm', N_("NUM"), MU_OPTION_DEFAULT,
    N_("limits nesting depth"),
    mu_c_int, &max_depth },
  { "dry-run", 'n', NULL, MU_OPTION_DEFAULT,
    N_("don't do anything: print what would be done"),
    mu_c_incr, &dry_run },
  { "verbose", 'v', NULL, MU_OPTION_DEFAULT,
    N_("verbosely print what is being done"),
    mu_c_incr, &verbose },
  { "trace", 't', NULL, MU_OPTION_DEFAULT,
    N_("sieve trace"),
    mu_c_bool, &action_trace },
  { "instructions", 'i', NULL, MU_OPTION_DEFAULT,
    N_("sieve instruction trace"),
    mu_c_void, NULL, cli_instr },
  { "keep", 'k', NULL, MU_OPTION_DEFAULT,
    N_("don't remove messages after archiving"),
    mu_c_bool, &keep_option },
  { "locus", 0, NULL, MU_OPTION_DEFAULT,
    N_("show action locations with verbose output"),
    mu_c_bool, &locus_option },
  { "statistics", 's', NULL, MU_OPTION_DEFAULT,
    N_("print statistics at the end of the run"),
    mu_c_bool, &stat_option },

  MU_OPTION_GROUP("Mailbox selection"),
  { "pattern", 'p', N_("PAT"), MU_OPTION_DEFAULT,
    N_("archive mailboxes matching pattern"),
    mu_c_string, &pattern, cli_pattern },
  { "pattern-type", 'T', N_("glob|imap|regex"), MU_OPTION_DEFAULT,
    N_("set type of the pattern used with -p and -x"),
    mu_c_bool, NULL, cli_pattern_type },
  { "exclude", 'x', N_("PAT"), MU_OPTION_DEFAULT,
    N_("add exclude pattern"),
    mu_c_string, &exclude, cli_pattern },

  MU_OPTION_GROUP("Sieve control"),
  { "require", 'r', N_("MODULE"), MU_OPTION_DEFAULT,
    N_("require MODULE in the produced Sieve script"),
    mu_c_string, NULL, cli_require },
  { "template-file", 'f', N_("FILE"), MU_OPTION_DEFAULT,
    N_("read template from FILE"),
    mu_c_string, &template_file },
  { "environment", 0, N_("NAME=VALUE"), MU_OPTION_DEFAULT,
    N_("set sieve environment value"),
    mu_c_string, NULL, cli_env },
  { "variable", 0, N_("NAME=VALUE"), MU_OPTION_DEFAULT,
    N_("set sieve variable"),
    mu_c_string, NULL, cli_var },
  { NULL }
}, *options[] = { mbar_options, NULL };

static struct mu_cli_setup cli = {
  .optv = options,
  .prog_doc = N_("mailbox archiver"),
  .prog_args = N_("DIR DEST [EXPR]"),
  .prog_extra_doc = N_("Sieve-specific debug levels:\n\
\n\
  trace1  -  print parse tree before optimization\n\
  trace2  -  print parse tree after optimization\n\
  trace3  -  print parser traces\n\
  trace4  -  print tests and actions being executed\n\
  trace9  -  print each Sieve instruction being executed\n")
};

static char *capa[] = {
  "debug",
  "mailbox",
  "locking",
  "logging",
  "sieve",
  NULL
};

int
main (int argc, char *argv[])
{
  struct mu_parseopt pohint;
  struct mu_cfg_parse_hints cfhint;
  mu_sieve_machine_t mach;
  int res;
  char *cond;

  MU_APP_INIT_NLS ();
  mu_register_local_mbox_formats ();
  mu_sieve_debug_init ();

  pohint.po_flags = 0;

  pohint.po_package_name = PACKAGE_NAME;
  pohint.po_flags |= MU_PARSEOPT_PACKAGE_NAME;

  pohint.po_package_url = PACKAGE_URL;
  pohint.po_flags |= MU_PARSEOPT_PACKAGE_URL;

  pohint.po_bug_address = PACKAGE_BUGREPORT;
  pohint.po_flags |= MU_PARSEOPT_BUG_ADDRESS;

  pohint.po_extra_info = mu_general_help_text;
  pohint.po_flags |= MU_PARSEOPT_EXTRA_INFO;

  pohint.po_version_hook = version_hook;
  pohint.po_flags |= MU_PARSEOPT_VERSION_HOOK;

  cfhint.site_file = mu_site_config_file ();
  cfhint.flags = MU_CFHINT_SITE_FILE | MU_CFHINT_PER_USER_FILE;

  mu_cli_capa_register (&mu_cli_capa_sieve);
  mu_cli_ext (argc, argv, &cli, &pohint, &cfhint, capa, NULL, &argc, &argv);
  if (dry_run)
    verbose++;

  if (mailbox_type)
    {
      int rc = mu_registrar_set_default_scheme (mailbox_type);
      switch (rc)
	{
	case 0:
	  break;

	case MU_ERR_NOENT:
	  abend (EX_USAGE, _("invalid mailbox scheme: %s"), mailbox_type);

	default:
	  abfun (EX_SOFTWARE, "mu_registrar_set_default_scheme",
		 mailbox_type, rc);
	}
    }

  switch (pattern_type)
    {
    case PAT_GLOB:
      patmat = glob_patmat;
      break;

    case PAT_IMAP:
      patmat = imap_patmat;
      break;

    case PAT_REGEX:
      list_regcomp (pattern);
      list_regcomp (exclude);
      patmat = regex_patmat;
    }

  if (template_file)
    read_template_file ();

  if (argc > 2)
    {
      char *cond;
      mu_argcv_join (argc - 2, argv + 2, " ", mu_argcv_escape_no, &cond);
      memlist_append (&cond_list, cond);
    }

  cond = build_cond ();

  build_requires ();
  mach = build_sieve (cond, (compile_only ? verbose : verbose > 2));
  free (cond);

  if (compile_only)
    {
      if (compile_only > 1)
	mu_sieve_disass (mach);
      exit (EX_OK);
    }

  if (argc < 2)
    abend (EX_USAGE, _("not enough arguments"));

  sourcedir = argv[0];
  destdir = argv[1];

  if (getuid () == 0)
    {
      switch_to_user = real_switch_to_user;
      switch_to_root = real_switch_to_root;
    }

  res = scan_folders (mach);
  mu_sieve_machine_destroy (&mach);
  return res;
}
