/* This file is part of gjson
   Copyright (C) 2020,2025 Sergey Poznyakoff

   Gjson is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   Gjson is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with gjson.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _JSON_H_
# define _JSON_H_
# include <stdlib.h>

enum json_value_type {
	json_null,
	json_bool,
	json_double,
	json_number = json_double,
	json_long,
	json_string,
	json_array,
	json_object
};

struct json_value;

struct json_array {
	size_t oc;
	struct json_value **ov;
	size_t on;
};

struct json_object {
	struct json_pair *head;
	struct json_pair *tail;
	size_t count;
};

struct json_value {
	enum json_value_type type;
	union {
		int b;                  /* json_bool */
		double n;               /* json_number */
		long i;                 /* json_long */
		char *s;                /* json_string */
		struct json_array *a;   /* json_array */
		struct json_object *o;  /* json_object */
	};
};

struct json_pair {
	struct json_pair *next;
	char *k;
	struct json_value *v;
};

struct json_format {
	size_t indent;
	int precision;
	void (*write) (void *, char const *, size_t);
	void *data;
};

int json_value_format(struct json_value *obj, struct json_format *fmt,
		       size_t level);
void json_value_free(struct json_value *);
int json_value_copy(struct json_value *val, struct json_value **new_val);

char const *json_strerror(int ec);

void *json_2nrealloc(void *p, size_t *pn, size_t s);

static inline enum json_value_type
json_type(struct json_value *obj)
{
	return obj->type;
}

static inline int
json_is_null(struct json_value *obj)
{
	return obj == NULL || obj->type == json_null;
}

static inline int
json_is_bool(struct json_value *obj)
{
	return obj != NULL && obj->type == json_bool;
}

static inline int
json_is_double(struct json_value *obj)
{
	return obj != NULL && obj->type == json_double;
}

static inline int
json_is_long(struct json_value *obj)
{
	return obj != NULL && obj->type == json_long;
}

static inline int
json_is_number(struct json_value *obj)
{
	return obj != NULL &&
		(obj->type == json_double || obj->type == json_long);
}

static inline int
json_is_string(struct json_value *obj)
{
	return obj != NULL && obj->type == json_string;
}

static inline int
json_is_array(struct json_value *obj)
{
	return obj != NULL && obj->type == json_array;
}

static inline int
json_is_object(struct json_value *obj)
{
	return obj != NULL && obj->type == json_object;
}

struct json_value *json_new_null(void);
struct json_value *json_new_bool(int b);
struct json_value *json_new_double(double n);
__attribute__ ((deprecated)) static inline struct json_value *
json_new_number(double n)
{
	return json_new_double(n);

}
struct json_value *json_new_long(long n);
struct json_value *json_new_string(char const *str);
struct json_value *json_new_object(void);
int json_object_set(struct json_value *obj, char const *name,
		    struct json_value *val);
int json_object_get(struct json_value *obj, char const *name,
		    struct json_value **retval);
int json_object_filter(struct json_value *obj,
		       int (*pred)(char const *, struct json_value *, void *),
		       void *data);

struct json_value *json_new_array(void);
static inline size_t json_array_length(struct json_value *j) {
	return j->a->oc;
}
int json_array_insert(struct json_value *j, size_t idx, struct json_value *v);
int json_array_append(struct json_value *j, struct json_value *v);
int json_array_set(struct json_value *j, size_t idx, struct json_value *v);
int json_array_get(struct json_value *j, size_t idx,
		   struct json_value **retval);


enum {
	JSON_E_NOERR,
	JSON_E_NOMEM,
	JSON_E_BADTOK,
	JSON_E_BADDELIM,
	JSON_E_BADSTRING,
	JSON_E_RANGE,
	JSON_E_EOF,
	JSON_E_USAGE
};

#define JSON_E_MOREINPUT 0x80

enum json_num_type {
	JSON_NUM_DOUBLE,
	JSON_NUM_LONG,
	JSON_NUM_AUTO
};
extern enum json_num_type json_num_type;

int json_parse(char const *input, size_t len,
	       struct json_value **retval, char **endp);
int json_parse_string(char const *input, struct json_value **retval,
		      char **endp);
typedef struct j_context *json_context;
int json_parse_incr(char const *input, size_t len, json_context *ctx,
		    struct json_value **retval, size_t *consumed, char **endp);

void json_context_free(json_context *ctx);

#endif
